/**
 * Client API pour l'application de logistique blockchain
 */

const API_BASE_URL = 'http://localhost:8000'; // À modifier en production

class APIClient {
    constructor() {
        this.token = localStorage.getItem('auth_token');
        this.user = JSON.parse(localStorage.getItem('user_data') || '{}');
    }

    // Méthodes d'authentification
    async login(username, password) {
        try {
            const response = await fetch(`${API_BASE_URL}/auth/token?username=${username}&password=${password}`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                }
            });

            if (!response.ok) {
                throw new Error('Identifiants incorrects');
            }

            const data = await response.json();
            this.token = data.access_token;
            this.user = data.user;

            // Stocker dans le localStorage
            localStorage.setItem('auth_token', this.token);
            localStorage.setItem('user_data', JSON.stringify(this.user));

            return data;
        } catch (error) {
            console.error('Login error:', error);
            throw error;
        }
    }

    logout() {
        localStorage.removeItem('auth_token');
        localStorage.removeItem('user_data');
        this.token = null;
        this.user = {};
        window.location.href = 'login.php';
    }

    // Headers avec authentification
    getHeaders() {
        const headers = {
            'Content-Type': 'application/json'
        };

        if (this.token) {
            headers['Authorization'] = `Bearer ${this.token}`;
        }

        return headers;
    }

    // Méthodes pour les expéditions
    async getDashboardStats() {
        return this.request('/dashboard/stats');
    }

    async getExpeditions(filters = {}) {
        const query = new URLSearchParams(filters).toString();
        return this.request(`/expeditions?${query}`);
    }

    async getExpedition(expeditionId) {
        return this.request(`/expeditions/${expeditionId}`);
    }

    async createExpedition(expeditionData) {
        return this.request('/expeditions', 'POST', expeditionData);
    }

    async updateExpedition(expeditionId, updates) {
        return this.request(`/expeditions/${expeditionId}`, 'PUT', updates);
    }

    async deleteExpedition(expeditionId) {
        return this.request(`/expeditions/${expeditionId}`, 'DELETE');
    }

    async addScan(scanData) {
        return this.request('/scans', 'POST', scanData);
    }

    async validateDelivery(validationData) {
        return this.request('/delivery/validate', 'POST', validationData);
    }

    // Méthodes pour les données IoT
    async sendIoTData(iotData) {
        return this.request('/iot/data', 'POST', iotData);
    }

    async getTemperatureHistory(expeditionId) {
        return this.request(`/iot/temperature/${expeditionId}`);
    }

    // Méthodes pour la blockchain
    async getBlockchainStatus() {
        return this.request('/blockchain/status');
    }

    async mineBlock(minerAddress) {
        return this.request('/blockchain/mine', 'POST', { miner_address: minerAddress });
    }

    async getBlockchainProof(expeditionId) {
        return this.request(`/blockchain/proof/${expeditionId}`);
    }

    async verifyOnBlockchain(proof) {
        return this.request('/blockchain/verify', 'POST', proof);
    }

    // Méthodes pour la traçabilité
    async getProductTraceability(productCode) {
        return this.request(`/traceability/${encodeURIComponent(productCode)}`);
    }

    async getActiveDeliveries() {
        return this.request('/map/deliveries');
    }

    // Méthode générique de requête
    async request(endpoint, method = 'GET', data = null) {
        const url = `${API_BASE_URL}${endpoint}`;
        const options = {
            method,
            headers: this.getHeaders()
        };

        if (data && (method === 'POST' || method === 'PUT' || method === 'PATCH')) {
            options.body = JSON.stringify(data);
        }

        try {
            const response = await fetch(url, options);

            if (response.status === 401) {
                // Token expiré, déconnexion
                this.logout();
                throw new Error('Session expirée');
            }

            if (!response.ok) {
                const errorData = await response.json().catch(() => ({}));
                throw new Error(errorData.detail || `HTTP ${response.status}`);
            }

            return await response.json();
        } catch (error) {
            console.error(`API request error: ${endpoint}`, error);
            throw error;
        }
    }
}

// Instance globale
const API = new APIClient();

// Fonctions utilitaires globales
function showAlert(title, message, type = 'error') {
    const alertDiv = document.createElement('div');
    alertDiv.className = `alert-notification ${type}`;
    alertDiv.innerHTML = `
        <div class="alert-icon ${type}">
            <i class="fas fa-${type === 'success' ? 'check-circle' : 'exclamation-triangle'}"></i>
        </div>
        <div class="alert-content">
            <div class="alert-title">${title}</div>
            <div class="alert-message">${message}</div>
        </div>
        <button class="alert-close" onclick="this.parentElement.remove()">&times;</button>
    `;

    document.body.appendChild(alertDiv);

    // Auto-remove après 5 secondes
    setTimeout(() => {
        if (alertDiv.parentElement) {
            alertDiv.remove();
        }
    }, 5000);
}

function showModal(modalId, options = {}) {
    const modal = document.getElementById(modalId) || createModal(modalId);
    
    if (options.title) {
        modal.querySelector('.modal-header h3').textContent = options.title;
    }
    
    if (options.content) {
        modal.querySelector('.modal-body').innerHTML = options.content;
    }
    
    if (options.buttons) {
        modal.querySelector('.modal-footer').innerHTML = options.buttons;
    }
    
    modal.style.display = 'flex';
}

function closeModal(modalId) {
    const modal = document.getElementById(modalId);
    if (modal) {
        modal.style.display = 'none';
    }
}

function createModal(modalId) {
    const modal = document.createElement('div');
    modal.id = modalId;
    modal.className = 'modal';
    modal.innerHTML = `
        <div class="modal-content">
            <div class="modal-header">
                <h3></h3>
                <button class="close-modal" onclick="closeModal('${modalId}')">&times;</button>
            </div>
            <div class="modal-body"></div>
            <div class="modal-footer"></div>
        </div>
    `;
    
    document.body.appendChild(modal);
    return modal;
}

// Exposer les fonctions globales
window.API = API;
window.showAlert = showAlert;
window.showModal = showModal;
window.closeModal = closeModal;