/**
 * Client WebSocket pour les mises à jour en temps réel
 */

class WebSocketClient {
    constructor() {
        this.ws = null;
        this.reconnectAttempts = 0;
        this.maxReconnectAttempts = 5;
        this.reconnectDelay = 1000;
        this.subscriptions = new Map();
        this.messageHandlers = [];
    }

    connect(expeditionId = null) {
        const wsUrl = expeditionId 
            ? `ws://localhost:8000/ws/${expeditionId}` 
            : 'ws://localhost:8000/ws/global';
        
        this.ws = new WebSocket(wsUrl);

        this.ws.onopen = () => {
            console.log('WebSocket connected');
            this.reconnectAttempts = 0;
            
            // Restaurer les abonnements
            this.subscriptions.forEach((callbacks, topic) => {
                this.subscribe(topic, callbacks);
            });
        };

        this.ws.onmessage = (event) => {
            const data = JSON.parse(event.data);
            this.handleMessage(data);
        };

        this.ws.onclose = () => {
            console.log('WebSocket disconnected');
            this.attemptReconnect();
        };

        this.ws.onerror = (error) => {
            console.error('WebSocket error:', error);
        };
    }

    subscribe(topic, callback) {
        if (!this.subscriptions.has(topic)) {
            this.subscriptions.set(topic, []);
        }
        
        this.subscriptions.get(topic).push(callback);
        
        // Envoyer un message d'abonnement si connecté
        if (this.ws && this.ws.readyState === WebSocket.OPEN) {
            this.ws.send(JSON.stringify({
                type: 'subscribe',
                topic: topic
            }));
        }
    }

    unsubscribe(topic, callback) {
        if (this.subscriptions.has(topic)) {
            const callbacks = this.subscriptions.get(topic);
            const index = callbacks.indexOf(callback);
            if (index > -1) {
                callbacks.splice(index, 1);
            }
        }
    }

    sendMessage(type, data) {
        if (this.ws && this.ws.readyState === WebSocket.OPEN) {
            this.ws.send(JSON.stringify({
                type,
                ...data
            }));
        }
    }

    handleMessage(data) {
        const { type, topic, ...messageData } = data;

        // Gérer les handlers spécifiques par type
        switch (type) {
            case 'status_update':
                this.handleStatusUpdate(messageData);
                break;
            case 'alert':
                this.handleAlert(messageData);
                break;
            case 'iot_data':
                this.handleIoTData(messageData);
                break;
            case 'scan':
                this.handleScan(messageData);
                break;
            default:
                // Gérer les handlers génériques
                this.messageHandlers.forEach(handler => {
                    if (handler.type === type || handler.type === '*') {
                        handler.callback(messageData);
                    }
                });
        }

        // Appeler les callbacks d'abonnement
        if (topic && this.subscriptions.has(topic)) {
            this.subscriptions.get(topic).forEach(callback => {
                callback(messageData);
            });
        }
    }

    handleStatusUpdate(data) {
        const { expedition_id, status } = data;
        
        // Mettre à jour l'interface utilisateur
        const statusElement = document.querySelector(`[data-expedition="${expedition_id}"] .expedition-status`);
        if (statusElement) {
            statusElement.textContent = status;
            statusElement.className = `status-badge status-${status.replace('_', '-')}`;
        }

        // Notification
        if (Notification.permission === 'granted') {
            new Notification(`Statut mis à jour - ${expedition_id}`, {
                body: `Nouveau statut: ${status}`,
                icon: '/assets/icons/truck.png'
            });
        }
    }

    handleAlert(data) {
        const { expedition_id, alert_type, severity, message } = data;
        
        // Afficher une alerte
        showAlert(`Alerte ${severity} - ${expedition_id}`, message, severity);
        
        // Mettre à jour le compteur d'alertes
        const alertCount = document.getElementById('notificationCount');
        if (alertCount) {
            const currentCount = parseInt(alertCount.textContent) || 0;
            alertCount.textContent = currentCount + 1;
            alertCount.style.display = 'block';
        }

        // Notification push
        if (Notification.permission === 'granted') {
            new Notification(`Alerte ${severity} - ${expedition_id}`, {
                body: message,
                icon: '/assets/icons/alert.png'
            });
        }
    }

    handleIoTData(data) {
        const { expedition_id, temperature, device_id } = data;
        
        // Mettre à jour l'affichage de température
        const tempElement = document.querySelector(`[data-expedition="${expedition_id}"] .current-temp`);
        if (tempElement) {
            tempElement.textContent = `${temperature}°C`;
            
            // Changer la couleur selon la température
            if (temperature > 8) {
                tempElement.className = 'temp-indicator temp-high';
            } else if (temperature < 2) {
                tempElement.className = 'temp-indicator temp-low';
            } else {
                tempElement.className = 'temp-indicator temp-ok';
            }
        }
    }

    handleScan(data) {
        const { expedition_id, etape, agent_id, timestamp } = data;
        
        // Ajouter à l'historique
        const historyElement = document.querySelector(`[data-expedition="${expedition_id}"] .expedition-history`);
        if (historyElement) {
            const scanItem = document.createElement('div');
            scanItem.className = 'timeline-item';
            scanItem.innerHTML = `
                <div class="timeline-content">
                    <div class="timeline-time">${new Date(timestamp).toLocaleTimeString()}</div>
                    <p class="timeline-text">Scan à l'étape ${etape} par ${agent_id}</p>
                </div>
            `;
            historyElement.appendChild(scanItem);
        }
    }

    attemptReconnect() {
        if (this.reconnectAttempts < this.maxReconnectAttempts) {
            this.reconnectAttempts++;
            
            setTimeout(() => {
                console.log(`Tentative de reconnexion ${this.reconnectAttempts}/${this.maxReconnectAttempts}`);
                this.connect();
            }, this.reconnectDelay * Math.pow(2, this.reconnectAttempts - 1));
        } else {
            console.error('Échec de reconnexion WebSocket');
            showAlert('Connexion perdue', 'La connexion en temps réel a été perdue. Actualisez la page.', 'warning');
        }
    }

    disconnect() {
        if (this.ws) {
            this.ws.close();
            this.ws = null;
        }
    }

    addMessageHandler(type, callback) {
        this.messageHandlers.push({ type, callback });
    }

    removeMessageHandler(callback) {
        const index = this.messageHandlers.findIndex(handler => handler.callback === callback);
        if (index > -1) {
            this.messageHandlers.splice(index, 1);
        }
    }
}

// Instance globale
const WSClient = new WebSocketClient();

// Fonction d'initialisation
function initializeWebSocket(expeditionId = null) {
    // Demander la permission pour les notifications
    if ('Notification' in window && Notification.permission === 'default') {
        Notification.requestPermission();
    }

    // Se connecter au WebSocket
    WSClient.connect(expeditionId);

    // Gérer la déconnexion
    window.addEventListener('beforeunload', () => {
        WSClient.disconnect();
    });
}

// Exposer globalement
window.WSClient = WSClient;
window.initializeWebSocket = initializeWebSocket;