<?php
/**
 * Système de notifications
 */

class NotificationSystem {
    private static $instance = null;
    private $notifications = [];
    
    private function __construct() {
        // Charger les notifications depuis la session
        if (isset($_SESSION['notifications'])) {
            $this->notifications = $_SESSION['notifications'];
        }
    }
    
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new NotificationSystem();
        }
        return self::$instance;
    }
    
    public function addNotification($type, $title, $message, $data = []) {
        $notification = [
            'id' => uniqid('notif_'),
            'type' => $type, // info, success, warning, error
            'title' => $title,
            'message' => $message,
            'data' => $data,
            'timestamp' => time(),
            'read' => false
        ];
        
        $this->notifications[] = $notification;
        $this->saveToSession();
        
        return $notification['id'];
    }
    
    public function getNotifications($unreadOnly = false, $limit = 50) {
        $notifications = $this->notifications;
        
        if ($unreadOnly) {
            $notifications = array_filter($notifications, function($n) {
                return !$n['read'];
            });
        }
        
        // Trier par date (plus récent en premier)
        usort($notifications, function($a, $b) {
            return $b['timestamp'] - $a['timestamp'];
        });
        
        return array_slice($notifications, 0, $limit);
    }
    
    public function markAsRead($notificationId) {
        foreach ($this->notifications as &$notification) {
            if ($notification['id'] === $notificationId) {
                $notification['read'] = true;
                break;
            }
        }
        $this->saveToSession();
    }
    
    public function markAllAsRead() {
        foreach ($this->notifications as &$notification) {
            $notification['read'] = true;
        }
        $this->saveToSession();
    }
    
    public function getUnreadCount() {
        return count(array_filter($this->notifications, function($n) {
            return !$n['read'];
        }));
    }
    
    public function clearOldNotifications($days = 30) {
        $cutoff = time() - ($days * 24 * 60 * 60);
        $this->notifications = array_filter($this->notifications, function($n) use ($cutoff) {
            return $n['timestamp'] > $cutoff;
        });
        $this->saveToSession();
    }
    
    private function saveToSession() {
        $_SESSION['notifications'] = $this->notifications;
    }
    
    // Méthodes d'usine pour les notifications courantes
    public static function createDeliveryNotification($expeditionId, $status, $message = '') {
        $instance = self::getInstance();
        
        $titles = [
            'created' => 'Nouvelle expédition créée',
            'scanned' => 'Expédition scannée',
            'in_transit' => 'Expédition en transit',
            'delivered' => 'Expédition livrée',
            'delayed' => 'Expédition retardée',
            'cancelled' => 'Expédition annulée'
        ];
        
        $defaultMessages = [
            'created' => "L'expédition $expeditionId a été créée",
            'scanned' => "L'expédition $expeditionId a été scannée",
            'in_transit' => "L'expédition $expeditionId est en cours de livraison",
            'delivered' => "L'expédition $expeditionId a été livrée avec succès",
            'delayed' => "L'expédition $expeditionId est retardée",
            'cancelled' => "L'expédition $expeditionId a été annulée"
        ];
        
        $title = $titles[$status] ?? 'Mise à jour expédition';
        $message = $message ?: ($defaultMessages[$status] ?? "Statut mis à jour pour l'expédition $expeditionId");
        
        return $instance->addNotification('info', $title, $message, [
            'expedition_id' => $expeditionId,
            'status' => $status
        ]);
    }
    
    public static function createAlertNotification($type, $expeditionId, $message, $severity = 'warning') {
        $instance = self::getInstance();
        
        $titles = [
            'temperature' => 'Alerte température',
            'humidity' => 'Alerte humidité',
            'shock' => 'Alerte choc détecté',
            'geofence' => 'Alerte zone de sortie',
            'delay' => 'Alerte retard',
            'blockchain' => 'Alerte blockchain'
        ];
        
        $title = $titles[$type] ?? 'Alerte système';
        
        return $instance->addNotification($severity, $title, $message, [
            'expedition_id' => $expeditionId,
            'alert_type' => $type,
            'severity' => $severity
        ]);
    }
    
    public static function createSystemNotification($title, $message, $type = 'info') {
        $instance = self::getInstance();
        return $instance->addNotification($type, $title, $message);
    }
}

// Initialiser le système de notifications
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

$notificationSystem = NotificationSystem::getInstance();
?>