/**
 * charts.js - Gestionnaire de graphiques pour la logistique
 */

class ChartManager {
    constructor() {
        this.charts = new Map(); // Map<chartId, chartInstance>
        this.chartColors = {
            primary: '#3498db',
            secondary: '#2ecc71',
            success: '#27ae60',
            warning: '#f39c12',
            danger: '#e74c3c',
            info: '#1abc9c',
            light: '#ecf0f1',
            dark: '#34495e',
            purple: '#9b59b6',
            orange: '#e67e22'
        };
        
        this.colorPalette = [
            '#3498db', '#2ecc71', '#e74c3c', '#f39c12', '#1abc9c',
            '#9b59b6', '#34495e', '#e67e22', '#95a5a6', '#16a085'
        ];
    }
    
    /**
     * Initialise un graphique de statut des expéditions
     * @param {string} canvasId - ID du canvas
     * @param {Object} data - Données du graphique
     */
    initializeStatusChart(canvasId, data = null) {
        const ctx = document.getElementById(canvasId);
        if (!ctx) {
            console.error(`Canvas ${canvasId} not found`);
            return null;
        }
        
        const defaultData = data || {
            labels: ['Préparation', 'En transit', 'En livraison', 'Livrée', 'Annulée', 'En alerte'],
            datasets: [{
                data: [12, 25, 15, 35, 3, 5],
                backgroundColor: [
                    this.chartColors.purple,      // Préparation
                    this.chartColors.primary,     // En transit
                    this.chartColors.info,        // En livraison
                    this.chartColors.success,     // Livrée
                    this.chartColors.dark,        // Annulée
                    this.chartColors.danger       // En alerte
                ],
                borderWidth: 2,
                borderColor: '#fff'
            }]
        };
        
        const chart = new Chart(ctx.getContext('2d'), {
            type: 'doughnut',
            data: defaultData,
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'right',
                        labels: {
                            padding: 20,
                            usePointStyle: true,
                            pointStyle: 'circle',
                            font: {
                                size: 12
                            }
                        }
                    },
                    tooltip: {
                        callbacks: {
                            label: function(context) {
                                const label = context.label || '';
                                const value = context.raw || 0;
                                const total = context.dataset.data.reduce((a, b) => a + b, 0);
                                const percentage = Math.round((value / total) * 100);
                                return `${label}: ${value} (${percentage}%)`;
                            }
                        }
                    }
                },
                cutout: '70%'
            }
        });
        
        this.charts.set(canvasId, chart);
        return chart;
    }
    
    /**
     * Initialise un graphique de performance mensuelle
     * @param {string} canvasId - ID du canvas
     * @param {Object} data - Données du graphique
     */
    initializePerformanceChart(canvasId, data = null) {
        const ctx = document.getElementById(canvasId);
        if (!ctx) return null;
        
        const defaultData = data || {
            labels: ['Jan', 'Fév', 'Mar', 'Avr', 'Mai', 'Jun', 'Jul', 'Aoû', 'Sep', 'Oct', 'Nov', 'Déc'],
            datasets: [
                {
                    label: 'Livraisons à l\'heure',
                    data: [92, 88, 95, 91, 94, 96, 98, 97, 95, 96, 94, 98],
                    borderColor: this.chartColors.success,
                    backgroundColor: 'rgba(39, 174, 96, 0.1)',
                    borderWidth: 3,
                    fill: true,
                    tension: 0.4
                },
                {
                    label: 'Température conforme',
                    data: [95, 90, 97, 93, 96, 98, 99, 98, 96, 97, 95, 99],
                    borderColor: this.chartColors.info,
                    backgroundColor: 'rgba(26, 188, 156, 0.1)',
                    borderWidth: 3,
                    fill: true,
                    tension: 0.4
                },
                {
                    label: 'Satisfaction client',
                    data: [88, 85, 90, 87, 92, 94, 96, 95, 93, 94, 92, 96],
                    borderColor: this.chartColors.warning,
                    backgroundColor: 'rgba(243, 156, 18, 0.1)',
                    borderWidth: 3,
                    fill: true,
                    tension: 0.4
                }
            ]
        };
        
        const chart = new Chart(ctx.getContext('2d'), {
            type: 'line',
            data: defaultData,
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'top',
                        labels: {
                            padding: 20,
                            font: {
                                size: 12
                            }
                        }
                    },
                    tooltip: {
                        mode: 'index',
                        intersect: false
                    }
                },
                scales: {
                    y: {
                        beginAtZero: false,
                        min: 80,
                        max: 100,
                        title: {
                            display: true,
                            text: 'Pourcentage (%)'
                        },
                        ticks: {
                            callback: function(value) {
                                return value + '%';
                            }
                        }
                    },
                    x: {
                        grid: {
                            display: false
                        }
                    }
                },
                interaction: {
                    intersect: false,
                    mode: 'nearest'
                }
            }
        });
        
        this.charts.set(canvasId, chart);
        return chart;
    }
    
    /**
     * Initialise un graphique de température
     * @param {string} canvasId - ID du canvas
     * @param {Array} temperatureData - Données de température
     */
    initializeTemperatureChart(canvasId, temperatureData = null) {
        const ctx = document.getElementById(canvasId);
        if (!ctx) return null;
        
        // Données par défaut si non fournies
        const defaultData = temperatureData || Array.from({length: 24}, (_, i) => ({
            hour: `${i.toString().padStart(2, '0')}:00`,
            temperature: 4 + Math.sin(i / 4) * 2 + Math.random() * 0.5,
            min: 2,
            max: 8
        }));
        
        const chart = new Chart(ctx.getContext('2d'), {
            type: 'line',
            data: {
                labels: defaultData.map(d => d.hour),
                datasets: [
                    {
                        label: 'Température actuelle',
                        data: defaultData.map(d => d.temperature),
                        borderColor: this.chartColors.primary,
                        backgroundColor: 'rgba(52, 152, 219, 0.1)',
                        borderWidth: 3,
                        pointRadius: 4,
                        pointBackgroundColor: this.chartColors.primary,
                        pointBorderColor: '#fff',
                        pointBorderWidth: 2,
                        tension: 0.4,
                        fill: false
                    },
                    {
                        label: 'Limite supérieure',
                        data: defaultData.map(d => d.max),
                        borderColor: this.chartColors.danger,
                        backgroundColor: 'rgba(231, 76, 60, 0.1)',
                        borderWidth: 2,
                        borderDash: [5, 5],
                        pointRadius: 0,
                        fill: '-1'
                    },
                    {
                        label: 'Limite inférieure',
                        data: defaultData.map(d => d.min),
                        borderColor: this.chartColors.warning,
                        backgroundColor: 'rgba(243, 156, 18, 0.1)',
                        borderWidth: 2,
                        borderDash: [5, 5],
                        pointRadius: 0,
                        fill: '-2'
                    }
                ]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'top',
                        labels: {
                            padding: 20,
                            font: {
                                size: 12
                            }
                        }
                    },
                    tooltip: {
                        callbacks: {
                            label: function(context) {
                                let label = context.dataset.label || '';
                                if (label) {
                                    label += ': ';
                                }
                                label += context.parsed.y.toFixed(1) + '°C';
                                return label;
                            }
                        }
                    }
                },
                scales: {
                    y: {
                        title: {
                            display: true,
                            text: 'Température (°C)'
                        },
                        suggestedMin: 0,
                        suggestedMax: 10
                    },
                    x: {
                        grid: {
                            display: false
                        }
                    }
                }
            }
        });
        
        this.charts.set(canvasId, chart);
        return chart;
    }
    
    /**
     * Initialise un graphique d'activité
     * @param {string} canvasId - ID du canvas
     * @param {Object} data - Données d'activité
     */
    initializeActivityChart(canvasId, data = null) {
        const ctx = document.getElementById(canvasId);
        if (!ctx) return null;
        
        const defaultData = data || {
            labels: ['Lun', 'Mar', 'Mer', 'Jeu', 'Ven', 'Sam', 'Dim'],
            datasets: [
                {
                    label: 'Scans',
                    data: [45, 52, 38, 67, 58, 22, 15],
                    backgroundColor: this.chartColors.primary,
                    borderRadius: 4
                },
                {
                    label: 'Livraisons',
                    data: [28, 35, 42, 55, 48, 18, 12],
                    backgroundColor: this.chartColors.success,
                    borderRadius: 4
                },
                {
                    label: 'Alertes',
                    data: [3, 5, 2, 7, 4, 1, 0],
                    backgroundColor: this.chartColors.danger,
                    borderRadius: 4
                }
            ]
        };
        
        const chart = new Chart(ctx.getContext('2d'), {
            type: 'bar',
            data: defaultData,
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'top',
                        labels: {
                            padding: 20,
                            font: {
                                size: 12
                            }
                        }
                    }
                },
                scales: {
                    x: {
                        stacked: false,
                        grid: {
                            display: false
                        }
                    },
                    y: {
                        stacked: false,
                        beginAtZero: true,
                        title: {
                            display: true,
                            text: 'Nombre'
                        }
                    }
                }
            }
        });
        
        this.charts.set(canvasId, chart);
        return chart;
    }
    
    /**
     * Initialise un graphique d'empreinte carbone
     * @param {string} canvasId - ID du canvas
     * @param {Object} data - Données carbone
     */
    initializeCarbonChart(canvasId, data = null) {
        const ctx = document.getElementById(canvasId);
        if (!ctx) return null;
        
        const defaultData = data || {
            labels: ['Transport routier', 'Transport froid', 'Logistique', 'Emballage', 'Compensation'],
            datasets: [{
                data: [45, 25, 15, 10, -30], // Compensation est négative
                backgroundColor: [
                    this.chartColors.primary,
                    this.chartColors.info,
                    this.chartColors.warning,
                    this.chartColors.purple,
                    this.chartColors.success
                ],
                borderWidth: 2,
                borderColor: '#fff'
            }]
        };
        
        const chart = new Chart(ctx.getContext('2d'), {
            type: 'pie',
            data: defaultData,
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'right',
                        labels: {
                            padding: 15,
                            usePointStyle: true,
                            pointStyle: 'circle',
                            font: {
                                size: 11
                            },
                            generateLabels: (chart) => {
                                const data = chart.data;
                                if (data.labels.length && data.datasets.length) {
                                    return data.labels.map((label, i) => {
                                        const value = data.datasets[0].data[i];
                                        const isNegative = value < 0;
                                        
                                        return {
                                            text: `${label}: ${isNegative ? '' : '+'}${value} kg CO₂`,
                                            fillStyle: data.datasets[0].backgroundColor[i],
                                            strokeStyle: '#fff',
                                            lineWidth: 2,
                                            hidden: false,
                                            index: i
                                        };
                                    });
                                }
                                return [];
                            }
                        }
                    },
                    tooltip: {
                        callbacks: {
                            label: function(context) {
                                const label = context.label || '';
                                const value = context.raw || 0;
                                const total = Math.abs(context.dataset.data.reduce((a, b) => a + Math.abs(b), 0));
                                const percentage = Math.round((Math.abs(value) / total) * 100);
                                const sign = value < 0 ? '-' : '+';
                                return `${label}: ${sign}${Math.abs(value)} kg CO₂ (${percentage}%)`;
                            }
                        }
                    }
                }
            }
        });
        
        this.charts.set(canvasId, chart);
        return chart;
    }
    
    /**
     * Initialise un graphique de jauge (gauge)
     * @param {string} canvasId - ID du canvas
     * @param {number} value - Valeur actuelle
     * @param {number} max - Valeur maximale
     * @param {string} label - Libellé
     */
    initializeGaugeChart(canvasId, value = 75, max = 100, label = 'Score') {
        const ctx = document.getElementById(canvasId);
        if (!ctx) return null;
        
        // Calcul des angles
        const percentage = (value / max) * 100;
        const angle = (percentage / 100) * Math.PI;
        
        // Déterminer la couleur
        let color;
        if (percentage >= 80) color = this.chartColors.success;
        else if (percentage >= 60) color = this.chartColors.warning;
        else color = this.chartColors.danger;
        
        const chart = new Chart(ctx.getContext('2d'), {
            type: 'doughnut',
            data: {
                datasets: [{
                    data: [value, max - value],
                    backgroundColor: [color, '#f0f0f0'],
                    borderWidth: 0,
                    circumference: 180, // Demi-cercle
                    rotation: 270, // Commence en haut
                    borderRadius: 10
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                cutout: '80%',
                plugins: {
                    legend: {
                        display: false
                    },
                    tooltip: {
                        enabled: false
                    }
                }
            },
            plugins: [{
                id: 'gaugeLabel',
                afterDraw: (chart) => {
                    const { ctx, chartArea: { width, height } } = chart;
                    const x = width / 2;
                    const y = height / 2 + 20;
                    
                    // Texte principal (valeur)
                    ctx.save();
                    ctx.font = 'bold 32px Arial';
                    ctx.fillStyle = color;
                    ctx.textAlign = 'center';
                    ctx.textBaseline = 'middle';
                    ctx.fillText(value.toString(), x, y - 20);
                    
                    // Libellé
                    ctx.font = '14px Arial';
                    ctx.fillStyle = '#666';
                    ctx.fillText(label, x, y + 20);
                    
                    // Pourcentage
                    ctx.font = '12px Arial';
                    ctx.fillStyle = '#999';
                    ctx.fillText(`${percentage.toFixed(1)}%`, x, y + 40);
                    
                    ctx.restore();
                }
            }]
        });
        
        this.charts.set(canvasId, chart);
        return chart;
    }
    
    /**
     * Initialise un graphique de tendance
     * @param {string} canvasId - ID du canvas
     * @param {Object} data - Données de tendance
     */
    initializeTrendChart(canvasId, data = null) {
        const ctx = document.getElementById(canvasId);
        if (!ctx) return null;
        
        const defaultData = data || {
            labels: ['Sem 1', 'Sem 2', 'Sem 3', 'Sem 4'],
            datasets: [
                {
                    label: 'Livraisons',
                    data: [120, 135, 128, 142],
                    borderColor: this.chartColors.primary,
                    backgroundColor: 'rgba(52, 152, 219, 0.1)',
                    borderWidth: 3,
                    fill: true,
                    tension: 0.4
                },
                {
                    label: 'Clients',
                    data: [45, 52, 48, 55],
                    borderColor: this.chartColors.success,
                    backgroundColor: 'rgba(46, 204, 113, 0.1)',
                    borderWidth: 3,
                    fill: true,
                    tension: 0.4
                }
            ]
        };
        
        const chart = new Chart(ctx.getContext('2d'), {
            type: 'line',
            data: defaultData,
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'top'
                    }
                },
                scales: {
                    y: {
                        beginAtZero: false,
                        grid: {
                            borderDash: [5, 5]
                        }
                    },
                    x: {
                        grid: {
                            display: false
                        }
                    }
                }
            }
        });
        
        this.charts.set(canvasId, chart);
        return chart;
    }
    
    /**
     * Met à jour un graphique existant
     * @param {string} chartId - ID du graphique
     * @param {Object} newData - Nouvelles données
     */
    updateChart(chartId, newData) {
        const chart = this.charts.get(chartId);
        if (!chart) {
            console.error(`Chart ${chartId} not found`);
            return;
        }
        
        chart.data = newData;
        chart.update();
    }
    
    /**
     * Met à jour les données d'un graphique
     * @param {string} chartId - ID du graphique
     * @param {Array} newData - Nouvelles données
     */
    updateChartData(chartId, newData) {
        const chart = this.charts.get(chartId);
        if (!chart) {
            console.error(`Chart ${chartId} not found`);
            return;
        }
        
        chart.data.datasets.forEach((dataset, index) => {
            if (newData[index]) {
                dataset.data = newData[index];
            }
        });
        
        chart.update();
    }
    
    /**
     * Ajoute un point de données à un graphique
     * @param {string} chartId - ID du graphique
     * @param {any} newValue - Nouvelle valeur
     * @param {string} datasetIndex - Index du dataset
     */
    addDataPoint(chartId, newValue, datasetIndex = 0) {
        const chart = this.charts.get(chartId);
        if (!chart) {
            console.error(`Chart ${chartId} not found`);
            return;
        }
        
        if (chart.data.datasets[datasetIndex]) {
            chart.data.datasets[datasetIndex].data.push(newValue);
            
            // Limiter à 50 points maximum
            if (chart.data.datasets[datasetIndex].data.length > 50) {
                chart.data.datasets[datasetIndex].data.shift();
            }
            
            // Ajouter un label si nécessaire
            if (chart.data.labels) {
                const now = new Date();
                const timeStr = now.toLocaleTimeString('fr-FR', { hour: '2-digit', minute: '2-digit' });
                chart.data.labels.push(timeStr);
                
                if (chart.data.labels.length > 50) {
                    chart.data.labels.shift();
                }
            }
            
            chart.update();
        }
    }
    
    /**
     * Exporte un graphique en image
     * @param {string} chartId - ID du graphique
     * @param {string} format - Format ('png', 'jpg', 'svg')
     */
    exportChart(chartId, format = 'png') {
        const chart = this.charts.get(chartId);
        if (!chart) {
            console.error(`Chart ${chartId} not found`);
            return;
        }
        
        const link = document.createElement('a');
        link.download = `graphique_${chartId}_${new Date().toISOString().split('T')[0]}.${format}`;
        link.href = chart.toBase64Image();
        link.click();
    }
    
    /**
     * Imprime un graphique
     * @param {string} chartId - ID du graphique
     */
    printChart(chartId) {
        const chart = this.charts.get(chartId);
        if (!chart) {
            console.error(`Chart ${chartId} not found`);
            return;
        }
        
        const printWindow = window.open('', '_blank');
        printWindow.document.write(`
            <html>
                <head>
                    <title>Graphique ${chartId}</title>
                    <style>
                        body { margin: 0; padding: 20px; }
                        img { max-width: 100%; height: auto; }
                        .chart-info { 
                            margin-top: 20px; 
                            font-family: Arial, sans-serif; 
                            color: #666;
                        }
                    </style>
                </head>
                <body>
                    <img src="${chart.toBase64Image()}">
                    <div class="chart-info">
                        <p>Exporté le: ${new Date().toLocaleString('fr-FR')}</p>
                        <p>ID du graphique: ${chartId}</p>
                    </div>
                </body>
            </html>
        `);
        printWindow.document.close();
        printWindow.print();
    }
    
    /**
     * Supprime un graphique
     * @param {string} chartId - ID du graphique
     */
    destroyChart(chartId) {
        const chart = this.charts.get(chartId);
        if (chart) {
            chart.destroy();
            this.charts.delete(chartId);
        }
    }
    
    /**
     * Supprime tous les graphiques
     */
    destroyAllCharts() {
        this.charts.forEach((chart, chartId) => {
            chart.destroy();
        });
        this.charts.clear();
    }
    
    /**
     * Crée un graphique personnalisé
     * @param {string} canvasId - ID du canvas
     * @param {string} type - Type de graphique
     * @param {Object} data - Données
     * @param {Object} options - Options
     */
    createCustomChart(canvasId, type, data, options = {}) {
        const ctx = document.getElementById(canvasId);
        if (!ctx) return null;
        
        const defaultOptions = {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: {
                    position: 'top'
                }
            }
        };
        
        const mergedOptions = this.deepMerge(defaultOptions, options);
        
        const chart = new Chart(ctx.getContext('2d'), {
            type: type,
            data: data,
            options: mergedOptions
        });
        
        this.charts.set(canvasId, chart);
        return chart;
    }
    
    /**
     * Fusionne profondément deux objets
     * @param {Object} target - Cible
     * @param {Object} source - Source
     * @returns {Object} Objet fusionné
     */
    deepMerge(target, source) {
        const output = Object.assign({}, target);
        
        if (this.isObject(target) && this.isObject(source)) {
            Object.keys(source).forEach(key => {
                if (this.isObject(source[key])) {
                    if (!(key in target)) {
                        Object.assign(output, { [key]: source[key] });
                    } else {
                        output[key] = this.deepMerge(target[key], source[key]);
                    }
                } else {
                    Object.assign(output, { [key]: source[key] });
                }
            });
        }
        
        return output;
    }
    
    /**
     * Vérifie si une valeur est un objet
     * @param {any} item - Valeur à vérifier
     * @returns {boolean} True si c'est un objet
     */
    isObject(item) {
        return (item && typeof item === 'object' && !Array.isArray(item));
    }
    
    /**
     * Génère des données aléatoires pour les démos
     * @param {number} count - Nombre de points
     * @param {number} min - Valeur minimale
     * @param {number} max - Valeur maximale
     * @returns {Array} Données aléatoires
     */
    generateRandomData(count = 10, min = 0, max = 100) {
        return Array.from({ length: count }, () => 
            Math.floor(Math.random() * (max - min + 1)) + min
        );
    }
    
    /**
     * Génère des labels de temps
     * @param {number} count - Nombre de labels
     * @param {string} unit - Unité ('hours', 'days', 'weeks', 'months')
     * @returns {Array} Labels de temps
     */
    generateTimeLabels(count = 10, unit = 'hours') {
        const now = new Date();
        const labels = [];
        
        for (let i = count - 1; i >= 0; i--) {
            const date = new Date(now);
            
            switch (unit) {
                case 'hours':
                    date.setHours(date.getHours() - i);
                    labels.push(date.toLocaleTimeString('fr-FR', { hour: '2-digit', minute: '2-digit' }));
                    break;
                case 'days':
                    date.setDate(date.getDate() - i);
                    labels.push(date.toLocaleDateString('fr-FR', { day: '2-digit', month: 'short' }));
                    break;
                case 'weeks':
                    date.setDate(date.getDate() - (i * 7));
                    labels.push(`Sem ${date.getWeek()}`);
                    break;
                case 'months':
                    date.setMonth(date.getMonth() - i);
                    labels.push(date.toLocaleDateString('fr-FR', { month: 'short', year: '2-digit' }));
                    break;
            }
        }
        
        return labels;
    }
}

// Extension de Date pour obtenir le numéro de semaine
Date.prototype.getWeek = function() {
    const date = new Date(this.getTime());
    date.setHours(0, 0, 0, 0);
    date.setDate(date.getDate() + 3 - (date.getDay() + 6) % 7);
    const week1 = new Date(date.getFullYear(), 0, 4);
    return 1 + Math.round(((date.getTime() - week1.getTime()) / 86400000 - 3 + (week1.getDay() + 6) % 7) / 7);
};

// Instance globale
let chartManager = null;

/**
 * Initialise le gestionnaire de graphiques
 * @returns {ChartManager} Instance du gestionnaire
 */
function initializeChartManager() {
    if (!chartManager) {
        chartManager = new ChartManager();
    }
    return chartManager;
}

/**
 * Initialise un graphique de statut
 */
function initializeStatusChart(canvasId, data) {
    if (!chartManager) {
        chartManager = new ChartManager();
    }
    return chartManager.initializeStatusChart(canvasId, data);
}

/**
 * Initialise un graphique de performance
 */
function initializePerformanceChart(canvasId, data) {
    if (!chartManager) {
        chartManager = new ChartManager();
    }
    return chartManager.initializePerformanceChart(canvasId, data);
}

/**
 * Initialise un graphique de température
 */
function initializeTemperatureChart(canvasId, data) {
    if (!chartManager) {
        chartManager = new ChartManager();
    }
    return chartManager.initializeTemperatureChart(canvasId, data);
}

/**
 * Initialise un graphique de jauge
 */
function initializeGaugeChart(canvasId, value, max, label) {
    if (!chartManager) {
        chartManager = new ChartManager();
    }
    return chartManager.initializeGaugeChart(canvasId, value, max, label);
}

/**
 * Met à jour un graphique
 */
function updateChart(chartId, newData) {
    if (!chartManager) return;
    chartManager.updateChart(chartId, newData);
}

/**
 * Exporte un graphique
 */
function exportChart(chartId, format) {
    if (!chartManager) return;
    chartManager.exportChart(chartId, format);
}

/**
 * Imprime un graphique
 */
function printChart(chartId) {
    if (!chartManager) return;
    chartManager.printChart(chartId);
}

/**
 * Initialise tous les graphiques du dashboard
 */
function initializeDashboardCharts() {
    const charts = {};
    
    // Graphique de statut
    const statusChart = initializeStatusChart('statusChart');
    if (statusChart) charts.statusChart = statusChart;
    
    // Graphique de performance
    const performanceChart = initializePerformanceChart('performanceChart');
    if (performanceChart) charts.performanceChart = performanceChart;
    
    // Graphique de température
    const temperatureChart = initializeTemperatureChart('temperatureChart');
    if (temperatureChart) charts.temperatureChart = temperatureChart;
    
    // Graphique d'activité
    const activityChart = initializeActivityChart('activityChart');
    if (activityChart) charts.activityChart = activityChart;
    
    // Graphique carbone
    const carbonChart = initializeCarbonChart('carbonChart');
    if (carbonChart) charts.carbonChart = carbonChart;
    
    // Jaunes
    const gauge1 = initializeGaugeChart('gauge1', 92, 100, 'Fiabilité');
    if (gauge1) charts.gauge1 = gauge1;
    
    const gauge2 = initializeGaugeChart('gauge2', 87, 100, 'Satisfaction');
    if (gauge2) charts.gauge2 = gauge2;
    
    const gauge3 = initializeGaugeChart('gauge3', 95, 100, 'Conformité');
    if (gauge3) charts.gauge3 = gauge3;
    
    return charts;
}

/**
 * Met à jour les graphiques avec des données réelles
 */
function updateChartsWithData(data) {
    if (!chartManager) return;
    
    // Mettre à jour le graphique de statut
    if (data.statusData && chartManager.charts.has('statusChart')) {
        updateChart('statusChart', data.statusData);
    }
    
    // Mettre à jour le graphique de performance
    if (data.performanceData && chartManager.charts.has('performanceChart')) {
        updateChart('performanceChart', data.performanceData);
    }
    
    // Mettre à jour le graphique de température
    if (data.temperatureData && chartManager.charts.has('temperatureChart')) {
        updateChart('temperatureChart', data.temperatureData);
    }
    
    // Ajouter des points de données en temps réel
    if (data.realtimeData) {
        Object.entries(data.realtimeData).forEach(([chartId, value]) => {
            if (chartManager.charts.has(chartId)) {
                chartManager.addDataPoint(chartId, value);
            }
        });
    }
}

// Exposer les fonctions globalement
window.initializeChartManager = initializeChartManager;
window.initializeStatusChart = initializeStatusChart;
window.initializePerformanceChart = initializePerformanceChart;
window.initializeTemperatureChart = initializeTemperatureChart;
window.initializeGaugeChart = initializeGaugeChart;
window.initializeDashboardCharts = initializeDashboardCharts;
window.updateChart = updateChart;
window.updateChartsWithData = updateChartsWithData;
window.exportChart = exportChart;
window.printChart = printChart;

// Initialiser automatiquement les graphiques quand la page est chargée
document.addEventListener('DOMContentLoaded', function() {
    // Initialiser le gestionnaire
    initializeChartManager();
    
    // Initialiser les graphiques du dashboard si on est sur dashboard.php
    if (document.getElementById('statusChart')) {
        setTimeout(() => {
            initializeDashboardCharts();
        }, 500);
    }
});