/**
 * Script principal de l'application
 */

document.addEventListener('DOMContentLoaded', function() {
    // Initialiser les composants communs
    initializeCommonComponents();
    
    // Vérifier l'authentification
    checkAuthentication();
    
    // Initialiser les notifications
    initializeNotifications();
    
    // Initialiser les tooltips
    initializeTooltips();
});

function initializeCommonComponents() {
    // Gestion des modales
    document.addEventListener('click', function(e) {
        if (e.target.classList.contains('modal')) {
            e.target.style.display = 'none';
        }
        
        if (e.target.classList.contains('close-modal')) {
            e.target.closest('.modal').style.display = 'none';
        }
    });
    
    // Gestion du menu mobile
    const menuToggle = document.getElementById('menuToggle');
    if (menuToggle) {
        menuToggle.addEventListener('click', function() {
            document.querySelector('.sidebar').classList.toggle('mobile-visible');
        });
    }
    
    // Formatage des dates
    formatAllDates();
    
    // Gestion des formulaires
    initializeForms();
}

function checkAuthentication() {
    const token = localStorage.getItem('auth_token');
    const user = localStorage.getItem('user_data');
    
    if (!token || !user) {
        // Rediriger vers la page de login si on n'est pas déjà dessus
        if (!window.location.href.includes('login.php')) {
            window.location.href = 'login.php?error=auth';
        }
    }
}

function initializeNotifications() {
    const notificationBell = document.getElementById('notificationBell');
    if (notificationBell) {
        notificationBell.addEventListener('click', function() {
            showNotifications();
        });
    }
    
    // Charger les notifications non lues
    loadUnreadNotifications();
}

function showNotifications() {
    // Récupérer les notifications
    API.request('/notifications').then(notifications => {
        const modalContent = `
            <div style="max-height: 400px; overflow-y: auto;">
                ${notifications.length > 0 ? 
                    notifications.map(notif => `
                        <div class="notification-item" style="padding: 15px; border-bottom: 1px solid #eee; cursor: pointer;"
                             onclick="viewNotification('${notif.id}')">
                            <div style="display: flex; justify-content: space-between;">
                                <div>
                                    <div style="font-weight: 600;">${notif.title}</div>
                                    <div style="font-size: 14px; color: #666; margin-top: 5px;">${notif.message}</div>
                                </div>
                                <div style="font-size: 12px; color: #999;">${formatTimeAgo(notif.timestamp)}</div>
                            </div>
                            ${notif.unread ? '<div style="width: 8px; height: 8px; background: #3498db; border-radius: 50%; margin-top: 10px;"></div>' : ''}
                        </div>
                    `).join('') :
                    '<div style="text-align: center; padding: 40px; color: #666;">Aucune notification</div>'
                }
            </div>
        `;
        
        showModal('notificationModal', {
            title: 'Notifications',
            content: modalContent
        });
    }).catch(error => {
        console.error('Error loading notifications:', error);
    });
}

function loadUnreadNotifications() {
    const notificationCount = document.getElementById('notificationCount');
    if (notificationCount) {
        API.request('/notifications/unread-count').then(data => {
            const count = data.count || 0;
            notificationCount.textContent = count;
            notificationCount.style.display = count > 0 ? 'block' : 'none';
        }).catch(() => {
            notificationCount.style.display = 'none';
        });
    }
}

function viewNotification(notificationId) {
    API.request(`/notifications/${notificationId}/read`, 'POST').then(() => {
        closeModal('notificationModal');
        loadUnreadNotifications();
    });
}

function initializeTooltips() {
    const tooltips = document.querySelectorAll('[data-tooltip]');
    tooltips.forEach(element => {
        element.addEventListener('mouseenter', showTooltip);
        element.addEventListener('mouseleave', hideTooltip);
    });
}

function showTooltip(e) {
    const tooltipText = e.target.getAttribute('data-tooltip');
    if (!tooltipText) return;
    
    const tooltip = document.createElement('div');
    tooltip.className = 'tooltip';
    tooltip.textContent = tooltipText;
    tooltip.style.position = 'absolute';
    tooltip.style.background = 'rgba(0,0,0,0.8)';
    tooltip.style.color = 'white';
    tooltip.style.padding = '5px 10px';
    tooltip.style.borderRadius = '4px';
    tooltip.style.fontSize = '12px';
    tooltip.style.zIndex = '10000';
    tooltip.style.pointerEvents = 'none';
    
    document.body.appendChild(tooltip);
    
    const rect = e.target.getBoundingClientRect();
    tooltip.style.top = (rect.top - tooltip.offsetHeight - 5) + 'px';
    tooltip.style.left = (rect.left + rect.width / 2 - tooltip.offsetWidth / 2) + 'px';
    
    e.target._tooltip = tooltip;
}

function hideTooltip(e) {
    if (e.target._tooltip) {
        e.target._tooltip.remove();
        delete e.target._tooltip;
    }
}

function formatAllDates() {
    document.querySelectorAll('[data-date]').forEach(element => {
        const dateString = element.getAttribute('data-date');
        if (dateString) {
            const date = new Date(dateString);
            const format = element.getAttribute('data-date-format') || 'datetime';
            
            let formattedDate;
            switch (format) {
                case 'date':
                    formattedDate = date.toLocaleDateString('fr-FR');
                    break;
                case 'time':
                    formattedDate = date.toLocaleTimeString('fr-FR');
                    break;
                case 'datetime':
                default:
                    formattedDate = date.toLocaleString('fr-FR');
                    break;
            }
            
            element.textContent = formattedDate;
        }
    });
}

function formatTimeAgo(dateString) {
    const date = new Date(dateString);
    const now = new Date();
    const diffMs = now - date;
    const diffSec = Math.floor(diffMs / 1000);
    const diffMin = Math.floor(diffSec / 60);
    const diffHour = Math.floor(diffMin / 60);
    const diffDay = Math.floor(diffHour / 24);
    
    if (diffDay > 7) {
        return date.toLocaleDateString('fr-FR');
    } else if (diffDay > 0) {
        return `il y a ${diffDay} jour${diffDay > 1 ? 's' : ''}`;
    } else if (diffHour > 0) {
        return `il y a ${diffHour} heure${diffHour > 1 ? 's' : ''}`;
    } else if (diffMin > 0) {
        return `il y a ${diffMin} minute${diffMin > 1 ? 's' : ''}`;
    } else {
        return 'à l\'instant';
    }
}

function initializeForms() {
    // Validation des formulaires
    document.querySelectorAll('form[data-validate]').forEach(form => {
        form.addEventListener('submit', function(e) {
            if (!validateForm(this)) {
                e.preventDefault();
                return false;
            }
        });
    });
}

function validateForm(form) {
    let isValid = true;
    const inputs = form.querySelectorAll('[required]');
    
    inputs.forEach(input => {
        if (!input.value.trim()) {
            markInvalid(input, 'Ce champ est obligatoire');
            isValid = false;
        } else {
            markValid(input);
        }
    });
    
    return isValid;
}

function markInvalid(input, message) {
    input.style.borderColor = '#e74c3c';
    input.parentElement.classList.add('error');
    
    // Ajouter un message d'erreur
    let errorMessage = input.parentElement.querySelector('.error-message');
    if (!errorMessage) {
        errorMessage = document.createElement('div');
        errorMessage.className = 'error-message';
        errorMessage.style.color = '#e74c3c';
        errorMessage.style.fontSize = '12px';
        errorMessage.style.marginTop = '5px';
        input.parentElement.appendChild(errorMessage);
    }
    
    errorMessage.textContent = message;
}

function markValid(input) {
    input.style.borderColor = '';
    input.parentElement.classList.remove('error');
    
    // Supprimer le message d'erreur
    const errorMessage = input.parentElement.querySelector('.error-message');
    if (errorMessage) {
        errorMessage.remove();
    }
}

// Fonctions utilitaires globales
function debounce(func, wait) {
    let timeout;
    return function executedFunction(...args) {
        const later = () => {
            clearTimeout(timeout);
            func(...args);
        };
        clearTimeout(timeout);
        timeout = setTimeout(later, wait);
    };
}

function throttle(func, limit) {
    let inThrottle;
    return function() {
        const args = arguments;
        const context = this;
        if (!inThrottle) {
            func.apply(context, args);
            inThrottle = true;
            setTimeout(() => inThrottle = false, limit);
        }
    };
}

// Exposer les utilitaires globalement
window.debounce = debounce;
window.throttle = throttle;