/**
 * map.js - Gestionnaire de carte interactive pour la logistique
 */

class MapManager {
    constructor(mapId = 'mainMap') {
        this.mapId = mapId;
        this.map = null;
        this.markers = new Map(); // Map<expedition_id, marker>
        this.routes = new Map(); // Map<expedition_id, polyline>
        this.heatmapLayer = null;
        this.clusterLayer = null;
        this.initialized = false;
        
        // Configuration
        this.config = {
            defaultCenter: [46.603354, 1.888334], // Centre France
            defaultZoom: 6,
            minZoom: 3,
            maxZoom: 18,
            tileLayerUrl: 'https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png',
            tileLayerAttribution: '© OpenStreetMap contributors',
            clusterOptions: {
                maxClusterRadius: 50,
                spiderfyOnMaxZoom: true,
                showCoverageOnHover: true
            }
        };
    }
    
    /**
     * Initialise la carte
     */
    initialize() {
        if (this.initialized) return this.map;
        
        try {
            // Créer la carte
            this.map = L.map(this.mapId).setView(
                this.config.defaultCenter, 
                this.config.defaultZoom
            );
            
            // Ajouter les tuiles OpenStreetMap
            L.tileLayer(this.config.tileLayerUrl, {
                attribution: this.config.tileLayerAttribution,
                minZoom: this.config.minZoom,
                maxZoom: this.config.maxZoom
            }).addTo(this.map);
            
            // Ajouter le contrôle de pleine écran
            L.control.fullscreen({
                position: 'topright',
                title: 'Plein écran',
                titleCancel: 'Quitter le plein écran'
            }).addTo(this.map);
            
            // Ajouter le contrôle de localisation
            L.control.locate({
                position: 'topright',
                drawCircle: true,
                follow: true,
                setView: true,
                keepCurrentZoomLevel: true,
                markerStyle: {
                    weight: 1,
                    opacity: 0.8,
                    fillOpacity: 0.8
                },
                circleStyle: {
                    weight: 1,
                    opacity: 0.2,
                    fillOpacity: 0.2
                },
                icon: 'fas fa-location-arrow',
                metric: true,
                strings: {
                    title: "Ma position"
                }
            }).addTo(this.map);
            
            // Ajouter le contrôle d'échelle
            L.control.scale({
                imperial: false,
                metric: true,
                position: 'bottomleft'
            }).addTo(this.map);
            
            // Initialiser le layer de clustering
            this.clusterLayer = L.markerClusterGroup(this.config.clusterOptions);
            this.map.addLayer(this.clusterLayer);
            
            this.initialized = true;
            console.log('Map initialized successfully');
            
            return this.map;
            
        } catch (error) {
            console.error('Error initializing map:', error);
            
            // Fallback : afficher un message d'erreur
            document.getElementById(this.mapId).innerHTML = `
                <div style="text-align: center; padding: 40px; color: #666;">
                    <i class="fas fa-exclamation-triangle" style="font-size: 48px; margin-bottom: 15px;"></i>
                    <h3>Erreur de chargement de la carte</h3>
                    <p>Impossible de charger la carte interactive. Vérifiez votre connexion internet.</p>
                    <button onclick="location.reload()" style="margin-top: 15px; padding: 10px 20px; 
                            background: #3498db; color: white; border: none; border-radius: 4px; cursor: pointer;">
                        <i class="fas fa-redo"></i> Réessayer
                    </button>
                </div>
            `;
            
            return null;
        }
    }
    
    /**
     * Met à jour les marqueurs des livraisons
     * @param {Array} deliveries - Liste des livraisons
     */
    updateDeliveries(deliveries) {
        if (!this.initialized || !this.map) return;
        
        // Supprimer les anciens marqueurs
        this.clearMarkers();
        
        if (!deliveries || deliveries.length === 0) {
            this.showNoDeliveriesMessage();
            return;
        }
        
        let bounds = L.latLngBounds();
        let hasValidMarkers = false;
        
        // Créer les nouveaux marqueurs
        deliveries.forEach(delivery => {
            if (delivery.position && delivery.position.lat && delivery.position.lon) {
                const marker = this.createDeliveryMarker(delivery);
                if (marker) {
                    this.markers.set(delivery.id, marker);
                    this.clusterLayer.addLayer(marker);
                    bounds.extend([delivery.position.lat, delivery.position.lon]);
                    hasValidMarkers = true;
                }
            }
        });
        
        // Ajuster la vue si on a des marqueurs valides
        if (hasValidMarkers && bounds.isValid()) {
            setTimeout(() => {
                this.map.fitBounds(bounds.pad(0.1));
            }, 100);
        } else {
            this.showNoValidPositionsMessage();
        }
        
        // Mettre à jour les statistiques
        this.updateMapStats(deliveries);
    }
    
    /**
     * Crée un marqueur pour une livraison
     * @param {Object} delivery - Données de la livraison
     * @returns {L.Marker} Marqueur Leaflet
     */
    createDeliveryMarker(delivery) {
        try {
            const { lat, lon } = delivery.position;
            
            // Déterminer la couleur et l'icône selon le statut
            const { color, icon } = this.getMarkerStyle(delivery);
            
            // Créer l'icône personnalisée
            const customIcon = L.divIcon({
                html: `
                    <div class="delivery-marker" 
                         style="
                            width: 40px;
                            height: 40px;
                            background: ${color};
                            border-radius: 50%;
                            display: flex;
                            align-items: center;
                            justify-content: center;
                            color: white;
                            font-weight: bold;
                            border: 3px solid white;
                            box-shadow: 0 2px 6px rgba(0,0,0,0.3);
                            cursor: pointer;
                         ">
                        <i class="fas ${icon}" style="font-size: 16px;"></i>
                    </div>
                `,
                className: 'delivery-marker-container',
                iconSize: [40, 40],
                iconAnchor: [20, 20]
            });
            
            // Créer le marqueur
            const marker = L.marker([lat, lon], { icon: customIcon });
            
            // Créer le popup
            const popupContent = this.createPopupContent(delivery);
            marker.bindPopup(popupContent, {
                maxWidth: 300,
                minWidth: 250,
                className: 'delivery-popup'
            });
            
            // Ajouter les événements
            marker.on('click', () => {
                this.onMarkerClick(delivery);
            });
            
            marker.on('mouseover', () => {
                marker.openPopup();
            });
            
            return marker;
            
        } catch (error) {
            console.error('Error creating marker:', error, delivery);
            return null;
        }
    }
    
    /**
     * Détermine le style du marqueur selon le statut
     * @param {Object} delivery - Données de la livraison
     * @returns {Object} Style {color, icon}
     */
    getMarkerStyle(delivery) {
        let color = '#3498db'; // Bleu par défaut
        let icon = 'fa-truck';
        
        if (delivery.alerte) {
            color = '#e74c3c'; // Rouge pour alerte
            icon = 'fa-exclamation-triangle';
        } else if (delivery.retard) {
            color = '#f39c12'; // Orange pour retard
            icon = 'fa-clock';
        } else if (delivery.statut === 'livree') {
            color = '#27ae60'; // Vert pour livrée
            icon = 'fa-check-circle';
        } else if (delivery.statut === 'preparation') {
            color = '#9b59b6'; // Violet pour préparation
            icon = 'fa-box';
        } else if (delivery.statut === 'en_transit') {
            color = '#3498db'; // Bleu pour transit
            icon = 'fa-shipping-fast';
        } else if (delivery.statut === 'en_livraison') {
            color = '#1abc9c'; // Turquoise pour livraison
            icon = 'fa-truck-loading';
        } else if (delivery.statut === 'annulee') {
            color = '#7f8c8d'; // Gris pour annulée
            icon = 'fa-times-circle';
        }
        
        return { color, icon };
    }
    
    /**
     * Crée le contenu du popup
     * @param {Object} delivery - Données de la livraison
     * @returns {string} HTML du popup
     */
    createPopupContent(delivery) {
        const statusClass = delivery.statut.replace('_', '-');
        
        return `
            <div class="delivery-popup-content">
                <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                    <h4 style="margin: 0; color: #2c3e50;">${delivery.id}</h4>
                    <span class="status-badge status-${statusClass}" style="font-size: 11px;">
                        ${delivery.statut}
                    </span>
                </div>
                
                <div style="margin: 10px 0;">
                    <div style="display: flex; align-items: center; gap: 8px; margin-bottom: 5px;">
                        <i class="fas fa-thermometer-half" style="color: #666; width: 16px;"></i>
                        <span style="font-weight: 600;">${delivery.temperature || 'N/A'}°C</span>
                        ${delivery.temperature_ok === false ? 
                            '<span style="color: #e74c3c; margin-left: 5px;"><i class="fas fa-exclamation-triangle"></i></span>' : 
                            ''
                        }
                    </div>
                    
                    <div style="display: flex; align-items: center; gap: 8px; margin-bottom: 5px;">
                        <i class="fas fa-road" style="color: #666; width: 16px;"></i>
                        <span>Progression: ${delivery.progression || 0}%</span>
                    </div>
                    
                    <div style="display: flex; align-items: center; gap: 8px; margin-bottom: 5px;">
                        <i class="fas fa-clock" style="color: #666; width: 16px;"></i>
                        <span>ETA: ${delivery.eta || 'N/A'}</span>
                    </div>
                    
                    ${delivery.retard ? `
                        <div style="color: #f39c12; margin: 5px 0; font-size: 12px;">
                            <i class="fas fa-exclamation-triangle"></i> En retard
                        </div>
                    ` : ''}
                    
                    ${delivery.alerte ? `
                        <div style="color: #e74c3c; margin: 5px 0; font-size: 12px;">
                            <i class="fas fa-exclamation-circle"></i> ${delivery.alerte_type || 'Alerte'}
                        </div>
                    ` : ''}
                </div>
                
                <div style="margin-top: 15px; text-align: center;">
                    <button onclick="window.open('expedition_detail.php?id=${delivery.id}', '_blank')"
                            style="background: #3498db; color: white; border: none; padding: 8px 15px; 
                                   border-radius: 4px; cursor: pointer; font-size: 12px; width: 100%;">
                        <i class="fas fa-external-link-alt"></i> Voir détails
                    </button>
                </div>
                
                <div style="margin-top: 10px; font-size: 10px; color: #999; text-align: center;">
                    Dernière mise à jour: ${new Date().toLocaleTimeString('fr-FR', {hour: '2-digit', minute: '2-digit'})}
                </div>
            </div>
        `;
    }
    
    /**
     * Gestion du clic sur un marqueur
     * @param {Object} delivery - Données de la livraison
     */
    onMarkerClick(delivery) {
        // Centrer la carte sur le marqueur
        this.map.setView([delivery.position.lat, delivery.position.lon], 13);
        
        // Émettre un événement personnalisé
        const event = new CustomEvent('deliveryMarkerClick', {
            detail: { delivery }
        });
        document.dispatchEvent(event);
        
        // Notification
        if (window.showAlert) {
            showAlert('Livraison sélectionnée', `Vous avez sélectionné ${delivery.id}`, 'info');
        }
    }
    
    /**
     * Trace un itinéraire entre deux points
     * @param {Object} start - Point de départ {lat, lon}
     * @param {Object} end - Point d'arrivée {lat, lon}
     * @param {string} expeditionId - ID de l'expédition
     */
    drawRoute(start, end, expeditionId) {
        if (!this.initialized || !this.map) return;
        
        // Supprimer l'ancien itinéraire s'il existe
        if (this.routes.has(expeditionId)) {
            this.map.removeLayer(this.routes.get(expeditionId));
            this.routes.delete(expeditionId);
        }
        
        // Créer la ligne
        const route = L.polyline([
            [start.lat, start.lon],
            [end.lat, end.lon]
        ], {
            color: '#3498db',
            weight: 3,
            opacity: 0.7,
            dashArray: '10, 10',
            className: 'delivery-route'
        }).addTo(this.map);
        
        this.routes.set(expeditionId, route);
        
        // Ajuster la vue pour inclure l'itinéraire
        const bounds = L.latLngBounds([start.lat, start.lon], [end.lat, end.lon]);
        this.map.fitBounds(bounds.pad(0.2));
        
        return route;
    }
    
    /**
     * Affiche une heatmap des livraisons
     * @param {Array} deliveries - Liste des livraisons
     */
    showHeatmap(deliveries) {
        if (!this.initialized || !this.map) return;
        
        // Nettoyer l'ancienne heatmap
        if (this.heatmapLayer) {
            this.map.removeLayer(this.heatmapLayer);
        }
        
        // Préparer les données pour la heatmap
        const heatmapData = deliveries
            .filter(d => d.position && d.position.lat && d.position.lon)
            .map(d => [d.position.lat, d.position.lon, 1]); // [lat, lon, intensity]
        
        if (heatmapData.length === 0) return;
        
        // Créer la heatmap (utiliser une bibliothèque comme Leaflet.heat si disponible)
        // Pour l'instant, on simule avec des cercles
        this.heatmapLayer = L.layerGroup().addTo(this.map);
        
        heatmapData.forEach(point => {
            L.circle([point[0], point[1]], {
                color: '#ff0000',
                fillColor: '#ff0000',
                fillOpacity: 0.2,
                radius: 10000 // 10km
            }).addTo(this.heatmapLayer);
        });
        
        return this.heatmapLayer;
    }
    
    /**
     * Affiche un message quand il n'y a pas de livraisons
     */
    showNoDeliveriesMessage() {
        const mapContainer = document.getElementById(this.mapId);
        if (!mapContainer) return;
        
        const existingMessage = mapContainer.querySelector('.no-deliveries-message');
        if (existingMessage) return;
        
        const messageDiv = document.createElement('div');
        messageDiv.className = 'no-deliveries-message';
        messageDiv.style.cssText = `
            position: absolute;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            background: rgba(255, 255, 255, 0.9);
            padding: 20px;
            border-radius: 10px;
            text-align: center;
            z-index: 1000;
            box-shadow: 0 4px 12px rgba(0,0,0,0.15);
            min-width: 300px;
        `;
        
        messageDiv.innerHTML = `
            <i class="fas fa-truck" style="font-size: 48px; color: #95a5a6; margin-bottom: 15px;"></i>
            <h3 style="margin: 0 0 10px; color: #2c3e50;">Aucune livraison active</h3>
            <p style="color: #666; margin: 0;">Aucune livraison n'est en cours pour le moment.</p>
        `;
        
        mapContainer.appendChild(messageDiv);
    }
    
    /**
     * Affiche un message quand il n'y a pas de positions valides
     */
    showNoValidPositionsMessage() {
        const mapContainer = document.getElementById(this.mapId);
        if (!mapContainer) return;
        
        const existingMessage = mapContainer.querySelector('.no-positions-message');
        if (existingMessage) return;
        
        const messageDiv = document.createElement('div');
        messageDiv.className = 'no-positions-message';
        messageDiv.style.cssText = `
            position: absolute;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            background: rgba(255, 255, 255, 0.9);
            padding: 20px;
            border-radius: 10px;
            text-align: center;
            z-index: 1000;
            box-shadow: 0 4px 12px rgba(0,0,0,0.15);
            min-width: 300px;
        `;
        
        messageDiv.innerHTML = `
            <i class="fas fa-map-marker-alt" style="font-size: 48px; color: #95a5a6; margin-bottom: 15px;"></i>
            <h3 style="margin: 0 0 10px; color: #2c3e50;">Positions non disponibles</h3>
            <p style="color: #666; margin: 0;">Les positions des livraisons ne sont pas disponibles pour le moment.</p>
        `;
        
        mapContainer.appendChild(messageDiv);
    }
    
    /**
     * Met à jour les statistiques de la carte
     * @param {Array} deliveries - Liste des livraisons
     */
    updateMapStats(deliveries) {
        const stats = {
            total: deliveries.length,
            withPosition: deliveries.filter(d => d.position && d.position.lat && d.position.lon).length,
            inTransit: deliveries.filter(d => d.statut === 'en_transit').length,
            inDelivery: deliveries.filter(d => d.statut === 'en_livraison').length,
            withAlerts: deliveries.filter(d => d.alerte).length,
            delayed: deliveries.filter(d => d.retard).length
        };
        
        // Émettre les statistiques
        const event = new CustomEvent('mapStatsUpdate', {
            detail: { stats }
        });
        document.dispatchEvent(event);
        
        // Mettre à jour l'interface si les éléments existent
        this.updateStatsDisplay(stats);
    }
    
    /**
     * Met à jour l'affichage des statistiques
     * @param {Object} stats - Statistiques
     */
    updateStatsDisplay(stats) {
        // Mettre à jour le compteur de livraisons actives
        const countElement = document.getElementById('activeDeliveriesCount');
        if (countElement) {
            countElement.textContent = stats.total;
        }
        
        // Mettre à jour les autres indicateurs
        const indicators = {
            'mapStatsWithPosition': stats.withPosition,
            'mapStatsInTransit': stats.inTransit,
            'mapStatsInDelivery': stats.inDelivery,
            'mapStatsWithAlerts': stats.withAlerts,
            'mapStatsDelayed': stats.delayed
        };
        
        Object.entries(indicators).forEach(([id, value]) => {
            const element = document.getElementById(id);
            if (element) {
                element.textContent = value;
            }
        });
    }
    
    /**
     * Centre la carte sur une position spécifique
     * @param {number} lat - Latitude
     * @param {number} lon - Longitude
     * @param {number} zoom - Niveau de zoom
     */
    centerOn(lat, lon, zoom = 13) {
        if (!this.initialized || !this.map) return;
        
        this.map.setView([lat, lon], zoom);
    }
    
    /**
     * Centre sur la position de l'utilisateur
     */
    centerOnUserLocation() {
        if (!navigator.geolocation) {
            if (window.showAlert) {
                showAlert('Erreur', 'La géolocalisation n\'est pas supportée par votre navigateur');
            }
            return;
        }
        
        navigator.geolocation.getCurrentPosition(
            (position) => {
                const { latitude, longitude } = position.coords;
                this.centerOn(latitude, longitude);
                
                // Ajouter un marqueur pour la position de l'utilisateur
                this.addUserMarker(latitude, longitude);
            },
            (error) => {
                console.error('Geolocation error:', error);
                if (window.showAlert) {
                    showAlert('Erreur', 'Impossible d\'obtenir votre position: ' + error.message);
                }
            },
            {
                enableHighAccuracy: true,
                timeout: 5000,
                maximumAge: 0
            }
        );
    }
    
    /**
     * Ajoute un marqueur pour la position de l'utilisateur
     * @param {number} lat - Latitude
     * @param {number} lon - Longitude
     */
    addUserMarker(lat, lon) {
        if (!this.initialized || !this.map) return;
        
        // Supprimer l'ancien marqueur s'il existe
        if (this.userMarker) {
            this.map.removeLayer(this.userMarker);
        }
        
        // Créer l'icône personnalisée
        const userIcon = L.divIcon({
            html: `
                <div style="
                    width: 30px;
                    height: 30px;
                    background: #e74c3c;
                    border-radius: 50%;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    color: white;
                    border: 3px solid white;
                    box-shadow: 0 2px 6px rgba(0,0,0,0.3);
                ">
                    <i class="fas fa-user" style="font-size: 14px;"></i>
                </div>
            `,
            className: 'user-marker',
            iconSize: [30, 30],
            iconAnchor: [15, 15]
        });
        
        // Créer le marqueur
        this.userMarker = L.marker([lat, lon], { icon: userIcon })
            .addTo(this.map)
            .bindPopup('Votre position')
            .openPopup();
    }
    
    /**
     * Nettoie tous les marqueurs
     */
    clearMarkers() {
        if (this.clusterLayer) {
            this.clusterLayer.clearLayers();
        }
        this.markers.clear();
        
        // Supprimer les messages
        const mapContainer = document.getElementById(this.mapId);
        if (mapContainer) {
            const messages = mapContainer.querySelectorAll('.no-deliveries-message, .no-positions-message');
            messages.forEach(msg => msg.remove());
        }
    }
    
    /**
     * Nettoie tous les itinéraires
     */
    clearRoutes() {
        this.routes.forEach(route => {
            if (this.map && route) {
                this.map.removeLayer(route);
            }
        });
        this.routes.clear();
    }
    
    /**
     * Nettoie complètement la carte
     */
    clearAll() {
        this.clearMarkers();
        this.clearRoutes();
        
        if (this.heatmapLayer) {
            this.map.removeLayer(this.heatmapLayer);
            this.heatmapLayer = null;
        }
    }
    
    /**
     * Actualise la carte
     */
    refresh() {
        if (!this.initialized || !this.map) return;
        
        // Forcer un redessin
        this.map.invalidateSize();
        
        // Émettre un événement de rafraîchissement
        const event = new CustomEvent('mapRefresh');
        document.dispatchEvent(event);
    }
    
    /**
     * Exporte la carte en image
     * @param {string} format - Format ('png', 'jpg')
     */
    exportMap(format = 'png') {
        if (!this.initialized || !this.map) return;
        
        // Utiliser html2canvas pour capturer la carte
        if (typeof html2canvas === 'undefined') {
            console.error('html2canvas non chargé');
            if (window.showAlert) {
                showAlert('Erreur', 'Fonction d\'export non disponible');
            }
            return;
        }
        
        const mapElement = document.getElementById(this.mapId);
        if (!mapElement) return;
        
        html2canvas(mapElement).then(canvas => {
            const link = document.createElement('a');
            link.download = `carte_livraisons_${new Date().toISOString().split('T')[0]}.${format}`;
            link.href = canvas.toDataURL(`image/${format}`);
            link.click();
        });
    }
    
    /**
     * Ajoute une couche géojson
     * @param {Object} geojson - Données GeoJSON
     * @param {Object} style - Style pour la couche
     */
    addGeoJSONLayer(geojson, style = {}) {
        if (!this.initialized || !this.map) return null;
        
        const defaultStyle = {
            color: '#3388ff',
            weight: 2,
            opacity: 0.7,
            fillOpacity: 0.2,
            ...style
        };
        
        const layer = L.geoJSON(geojson, {
            style: defaultStyle,
            onEachFeature: (feature, layer) => {
                if (feature.properties && feature.properties.name) {
                    layer.bindPopup(feature.properties.name);
                }
            }
        }).addTo(this.map);
        
        return layer;
    }
}

// Instance globale
let mapManager = null;

/**
 * Initialise la carte
 * @param {string} mapId - ID de l'élément carte
 * @returns {MapManager} Instance du gestionnaire de carte
 */
function initializeMap(mapId = 'mainMap') {
    if (!mapManager) {
        mapManager = new MapManager(mapId);
    }
    
    // Attendre que la carte soit rendue
    setTimeout(() => {
        mapManager.initialize();
    }, 100);
    
    return mapManager;
}

/**
 * Charge les livraisons sur la carte
 * @param {Array} deliveries - Liste des livraisons
 */
function loadDeliveriesOnMap(deliveries) {
    if (!mapManager) {
        console.error('Map not initialized');
        return;
    }
    
    mapManager.updateDeliveries(deliveries);
}

/**
 * Centre sur une position spécifique
 */
function centerMapOn(lat, lon, zoom) {
    if (!mapManager) return;
    mapManager.centerOn(lat, lon, zoom);
}

/**
 * Centre sur la position de l'utilisateur
 */
function centerOnMyLocation() {
    if (!mapManager) return;
    mapManager.centerOnUserLocation();
}

/**
 * Rafraîchit la carte
 */
function refreshMap() {
    if (!mapManager) return;
    mapManager.refresh();
}

/**
 * Exporte la carte
 */
function exportMap(format) {
    if (!mapManager) return;
    mapManager.exportMap(format);
}

// Exposer les fonctions globalement
window.initializeMap = initializeMap;
window.loadDeliveriesOnMap = loadDeliveriesOnMap;
window.centerMapOn = centerMapOn;
window.centerOnMyLocation = centerOnMyLocation;
window.refreshMap = refreshMap;
window.exportMap = exportMap;