/**
 * Script spécifique pour la page de profil
 */

document.addEventListener('DOMContentLoaded', function() {
    // Initialiser les graphiques de statistiques
    initializeProfileCharts();
    
    // Initialiser les tooltips personnalisés
    initializeProfileTooltips();
    
    // Vérifier l'expiration du mot de passe
    checkPasswordExpiry();
});

function initializeProfileCharts() {
    // Graphique des performances
    const performanceCtx = document.getElementById('performanceChart');
    if (performanceCtx) {
        new Chart(performanceCtx.getContext('2d'), {
            type: 'line',
            data: {
                labels: ['Jan', 'Fév', 'Mar', 'Avr', 'Mai', 'Jun', 'Jul', 'Aoû', 'Sep', 'Oct', 'Nov', 'Déc'],
                datasets: [{
                    label: 'Performance',
                    data: [85, 88, 90, 87, 92, 94, 95, 93, 96, 97, 95, 98],
                    borderColor: '#3498db',
                    backgroundColor: 'rgba(52, 152, 219, 0.1)',
                    borderWidth: 2,
                    fill: true,
                    tension: 0.4
                }]
            },
            options: {
                responsive: true,
                plugins: {
                    legend: {
                        display: false
                    }
                },
                scales: {
                    y: {
                        beginAtZero: false,
                        min: 80,
                        max: 100
                    }
                }
            }
        });
    }
    
    // Graphique de répartition des activités
    const activityCtx = document.getElementById('activityChart');
    if (activityCtx) {
        new Chart(activityCtx.getContext('2d'), {
            type: 'doughnut',
            data: {
                labels: ['Livraisons', 'Scans', 'Validations', 'Audits', 'Rapports'],
                datasets: [{
                    data: [40, 25, 20, 10, 5],
                    backgroundColor: [
                        '#3498db',
                        '#2ecc71',
                        '#f39c12',
                        '#9b59b6',
                        '#e74c3c'
                    ],
                    borderWidth: 2,
                    borderColor: '#fff'
                }]
            },
            options: {
                responsive: true,
                plugins: {
                    legend: {
                        position: 'bottom'
                    }
                }
            }
        });
    }
}

function initializeProfileTooltips() {
    // Tooltips pour les statistiques
    document.querySelectorAll('.stat-item').forEach(item => {
        item.addEventListener('mouseenter', function() {
            const tooltip = this.getAttribute('data-tooltip');
            if (tooltip) {
                showCustomTooltip(this, tooltip);
            }
        });
        
        item.addEventListener('mouseleave', function() {
            hideCustomTooltip();
        });
    });
}

function showCustomTooltip(element, message) {
    const tooltip = document.createElement('div');
    tooltip.className = 'custom-tooltip';
    tooltip.textContent = message;
    tooltip.style.position = 'fixed';
    tooltip.style.background = 'rgba(0,0,0,0.9)';
    tooltip.style.color = 'white';
    tooltip.style.padding = '8px 12px';
    tooltip.style.borderRadius = '4px';
    tooltip.style.fontSize = '12px';
    tooltip.style.zIndex = '10000';
    tooltip.style.maxWidth = '300px';
    tooltip.style.pointerEvents = 'none';
    
    document.body.appendChild(tooltip);
    
    const rect = element.getBoundingClientRect();
    tooltip.style.top = (rect.top - tooltip.offsetHeight - 10) + 'px';
    tooltip.style.left = (rect.left + rect.width / 2 - tooltip.offsetWidth / 2) + 'px';
    
    window.currentTooltip = tooltip;
}

function hideCustomTooltip() {
    if (window.currentTooltip) {
        window.currentTooltip.remove();
        window.currentTooltip = null;
    }
}

function checkPasswordExpiry() {
    // Vérifier si le mot de passe expire bientôt
    const lastPasswordChange = localStorage.getItem('last_password_change');
    if (lastPasswordChange) {
        const daysSinceChange = Math.floor((Date.now() - parseInt(lastPasswordChange)) / (1000 * 60 * 60 * 24));
        
        if (daysSinceChange > 85) { // Avertir à partir de 85 jours
            showPasswordExpiryWarning(daysSinceChange);
        }
    }
}

function showPasswordExpiryWarning(days) {
    const daysLeft = 90 - days;
    
    const warningDiv = document.createElement('div');
    warningDiv.className = 'alert alert-warning';
    warningDiv.style.position = 'fixed';
    warningDiv.style.bottom = '20px';
    warningDiv.style.right = '20px';
    warningDiv.style.maxWidth = '400px';
    warningDiv.style.zIndex = '10000';
    warningDiv.innerHTML = `
        <div style="display: flex; align-items: flex-start; gap: 15px;">
            <div style="flex-shrink: 0;">
                <i class="fas fa-exclamation-triangle" style="font-size: 24px;"></i>
            </div>
            <div>
                <div style="font-weight: 600; margin-bottom: 5px;">Mot de passe expirant bientôt</div>
                <div style="font-size: 14px;">
                    Votre mot de passe a été changé il y a ${days} jours. 
                    ${daysLeft > 0 ? `Il expirera dans ${daysLeft} jour${daysLeft > 1 ? 's' : ''}.` : 'Il a expiré.'}
                </div>
                <button onclick="changePasswordNow()" 
                        style="margin-top: 10px; background: #f39c12; color: white; 
                               border: none; padding: 8px 16px; border-radius: 4px; 
                               cursor: pointer; font-weight: 600;">
                    Changer maintenant
                </button>
                <button onclick="this.parentElement.parentElement.parentElement.remove()" 
                        style="margin-top: 10px; margin-left: 10px; background: none; 
                               border: none; color: #666; cursor: pointer;">
                    Rappeler plus tard
                </button>
            </div>
        </div>
    `;
    
    document.body.appendChild(warningDiv);
}

function changePasswordNow() {
    // Ouvrir le formulaire de changement de mot de passe
    document.getElementById('passwordForm').scrollIntoView({ behavior: 'smooth' });
    
    // Supprimer l'avertissement
    const warning = document.querySelector('.alert-warning[style*="position: fixed"]');
    if (warning) {
        warning.remove();
    }
}

// Fonction pour exporter les données du profil
function exportProfileData(format = 'json') {
    const userData = {
        profile: <?php echo json_encode($user); ?>,
        stats: <?php echo json_encode($user_stats); ?>,
        export_date: new Date().toISOString()
    };
    
    let content, mimeType, filename;
    
    switch (format) {
        case 'json':
            content = JSON.stringify(userData, null, 2);
            mimeType = 'application/json';
            filename = `profile_${userData.profile.id}_${new Date().toISOString().split('T')[0]}.json`;
            break;
            
        case 'csv':
            content = convertToCSV(userData);
            mimeType = 'text/csv';
            filename = `profile_${userData.profile.id}_${new Date().toISOString().split('T')[0]}.csv`;
            break;
            
        default:
            console.error('Format non supporté:', format);
            return;
    }
    
    // Créer et télécharger le fichier
    const blob = new Blob([content], { type: mimeType });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = filename;
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    URL.revokeObjectURL(url);
}

function convertToCSV(data) {
    // Conversion simplifiée pour l'exemple
    let csv = 'Catégorie,Clé,Valeur\n';
    
    // Profil
    for (const [key, value] of Object.entries(data.profile)) {
        if (typeof value === 'string' || typeof value === 'number') {
            csv += `Profil,${key},${value}\n`;
        }
    }
    
    // Statistiques
    for (const [key, value] of Object.entries(data.stats)) {
        csv += `Statistiques,${key},${value}\n`;
    }
    
    return csv;
}

// Exposer les fonctions globalement
window.exportProfileData = exportProfileData;
window.showPasswordExpiryWarning = showPasswordExpiryWarning;