<?php
$page_title = "Carte des livraisons";
$page_scripts = ['js/map.js'];
include 'includes/header.php';
?>

<div class="page-header">
    <h1><i class="fas fa-map-marked-alt"></i> Carte des livraisons en direct</h1>
    <div style="display: flex; gap: 10px;">
        <button class="btn" onclick="refreshMap()">
            <i class="fas fa-sync-alt"></i> Actualiser
        </button>
        <button class="btn btn-success" onclick="centerOnMyLocation()">
            <i class="fas fa-location-arrow"></i> Ma position
        </button>
    </div>
</div>

<div class="grid grid-2">
    <!-- Carte principale -->
    <div class="card">
        <div class="card-header">
            <h2 class="card-title">Livraisons actives</h2>
            <div style="display: flex; align-items: center; gap: 15px;">
                <div class="legend">
                    <div style="display: flex; align-items: center; gap: 5px;">
                        <div style="width: 15px; height: 15px; background: #27ae60; border-radius: 50%;"></div>
                        <span style="font-size: 12px;">En temps</span>
                    </div>
                    <div style="display: flex; align-items: center; gap: 5px;">
                        <div style="width: 15px; height: 15px; background: #f39c12; border-radius: 50%;"></div>
                        <span style="font-size: 12px;">En retard</span>
                    </div>
                    <div style="display: flex; align-items: center; gap: 5px;">
                        <div style="width: 15px; height: 15px; background: #e74c3c; border-radius: 50%;"></div>
                        <span style="font-size: 12px;">Alerte</span>
                    </div>
                </div>
            </div>
        </div>
        <div class="map-container" id="mainMap"></div>
        <div style="padding: 15px; background: #f8f9fa; border-top: 1px solid #eee;">
            <div style="display: flex; justify-content: space-between; align-items: center;">
                <div>
                    <span id="activeDeliveriesCount">0</span> livraisons actives
                </div>
                <div style="font-size: 12px; color: #666;">
                    Dernière mise à jour: <span id="lastUpdate">-</span>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Liste des livraisons -->
    <div class="card">
        <div class="card-header">
            <h2 class="card-title">Détails des livraisons</h2>
            <div class="form-group" style="margin: 0; width: 200px;">
                <select id="filterStatus" onchange="filterDeliveries()">
                    <option value="all">Tous les statuts</option>
                    <option value="en_transit">En transit</option>
                    <option value="en_livraison">En livraison</option>
                    <option value="en_alerte">En alerte</option>
                </select>
            </div>
        </div>
        <div style="max-height: 500px; overflow-y: auto;">
            <div id="deliveriesList">
                <!-- Rempli par JavaScript -->
            </div>
        </div>
    </div>
</div>

<!-- Statistiques de température -->
<div class="card">
    <div class="card-header">
        <h2 class="card-title">Surveillance de température</h2>
    </div>
    <div class="grid grid-3">
        <div style="text-align: center; padding: 20px;">
            <div class="temp-indicator temp-ok">
                <i class="fas fa-check-circle"></i>
                <span id="tempOkCount">0</span>
            </div>
            <div style="margin-top: 10px; font-size: 14px;">Dans les normes</div>
        </div>
        <div style="text-align: center; padding: 20px;">
            <div class="temp-indicator temp-high">
                <i class="fas fa-exclamation-triangle"></i>
                <span id="tempHighCount">0</span>
            </div>
            <div style="margin-top: 10px; font-size: 14px;">Trop élevée</div>
        </div>
        <div style="text-align: center; padding: 20px;">
            <div class="temp-indicator temp-low">
                <i class="fas fa-exclamation-triangle"></i>
                <span id="tempLowCount">0</span>
            </div>
            <div style="margin-top: 10px; font-size: 14px;">Trop basse</div>
        </div>
    </div>
</div>

<script>
let map;
let markers = [];
let deliveriesData = [];

document.addEventListener('DOMContentLoaded', function() {
    initializeMap();
    loadDeliveries();
    initializeWebSocket();
    setInterval(loadDeliveries, 30000); // Rafraîchir toutes les 30 secondes
});

function initializeMap() {
    // Initialiser la carte centrée sur la France
    map = L.map('mainMap').setView([46.603354, 1.888334], 6);
    
    // Ajouter les tuiles OpenStreetMap
    L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
        attribution: '© OpenStreetMap contributors'
    }).addTo(map);
    
    // Ajouter le contrôle de pleine écran
    map.addControl(new L.Control.Fullscreen());
}

function loadDeliveries() {
    API.getActiveDeliveries().then(data => {
        deliveriesData = data.deliveries;
        updateDeliveriesList(data.deliveries);
        updateMapMarkers(data.deliveries);
        updateTemperatureStats(data.temperature_stats);
        updateLastUpdate();
    }).catch(error => {
        showAlert('Erreur', 'Impossible de charger les livraisons');
    });
}

function updateMapMarkers(deliveries) {
    // Supprimer les anciens marqueurs
    markers.forEach(marker => map.removeLayer(marker));
    markers = [];
    
    deliveries.forEach(delivery => {
        if (delivery.position && delivery.position.lat && delivery.position.lon) {
            // Déterminer la couleur en fonction du statut
            let color = '#3498db'; // Bleu par défaut
            
            if (delivery.alerte) {
                color = '#e74c3c'; // Rouge pour alerte
            } else if (delivery.retard) {
                color = '#f39c12'; // Orange pour retard
            } else if (delivery.statut === 'livree') {
                color = '#27ae60'; // Vert pour livrée
            }
            
            // Créer une icône personnalisée
            const icon = L.divIcon({
                html: `
                    <div style="
                        width: 30px;
                        height: 30px;
                        background: ${color};
                        border-radius: 50%;
                        display: flex;
                        align-items: center;
                        justify-content: center;
                        color: white;
                        font-weight: bold;
                        border: 2px solid white;
                        box-shadow: 0 2px 4px rgba(0,0,0,0.3);
                    ">
                        <i class="fas fa-truck"></i>
                    </div>
                `,
                className: 'delivery-marker',
                iconSize: [30, 30],
                iconAnchor: [15, 15]
            });
            
            // Ajouter le marqueur
            const marker = L.marker([delivery.position.lat, delivery.position.lon], { icon })
                .addTo(map)
                .bindPopup(`
                    <div style="min-width: 200px;">
                        <h4 style="margin: 0 0 10px;">${delivery.id}</h4>
                        <p><strong>Statut:</strong> ${delivery.statut}</p>
                        <p><strong>Température:</strong> ${delivery.temperature}°C</p>
                        <p><strong>Progression:</strong> ${delivery.progression}%</p>
                        ${delivery.alerte ? `<p style="color: #e74c3c;"><strong>Alerte:</strong> ${delivery.alerte_type}</p>` : ''}
                        <div style="margin-top: 10px; text-align: center;">
                            <button onclick="viewExpedition('${delivery.id}')" 
                                    style="background: #3498db; color: white; border: none; padding: 5px 10px; border-radius: 3px; cursor: pointer;">
                                Voir détails
                            </button>
                        </div>
                    </div>
                `);
            
            markers.push(marker);
        }
    });
    
    // Ajuster la vue de la carte pour inclure tous les marqueurs
    if (markers.length > 0) {
        const group = new L.featureGroup(markers);
        map.fitBounds(group.getBounds().pad(0.1));
    }
}

function updateDeliveriesList(deliveries) {
    document.getElementById('activeDeliveriesCount').textContent = deliveries.length;
    
    if (deliveries.length === 0) {
        document.getElementById('deliveriesList').innerHTML = `
            <div style="text-align: center; padding: 40px; color: #666;">
                <i class="fas fa-truck" style="font-size: 48px; margin-bottom: 15px;"></i>
                <p>Aucune livraison active pour le moment</p>
            </div>
        `;
        return;
    }
    
    let html = '';
    deliveries.forEach(delivery => {
        const statusClass = delivery.statut.replace('_', '-');
        const isLate = delivery.retard ? '<span style="color: #f39c12; margin-left: 10px;"><i class="fas fa-clock"></i> En retard</span>' : '';
        const hasAlert = delivery.alerte ? '<span style="color: #e74c3c; margin-left: 10px;"><i class="fas fa-exclamation-triangle"></i> Alerte</span>' : '';
        
        html += `
            <div class="expedition-item" onclick="viewExpedition('${delivery.id}')" 
                 style="cursor: pointer; ${delivery.alerte ? 'border-left: 4px solid #e74c3c;' : ''}">
                <div class="expedition-info">
                    <div class="expedition-icon">
                        <i class="fas fa-truck"></i>
                    </div>
                    <div class="expedition-details">
                        <h4>${delivery.id}</h4>
                        <div style="display: flex; align-items: center; gap: 10px; margin-top: 5px;">
                            <span class="status-badge status-${statusClass}">${delivery.statut}</span>
                            ${isLate}
                            ${hasAlert}
                        </div>
                        <div class="expedition-meta">
                            <span><i class="fas fa-thermometer-half"></i> ${delivery.temperature}°C</span>
                            <span><i class="fas fa-road"></i> ${delivery.progression}%</span>
                            <span><i class="fas fa-clock"></i> ${delivery.eta}</span>
                        </div>
                    </div>
                </div>
                <div class="expedition-actions">
                    <button class="btn" style="padding: 5px 10px; font-size: 12px;" 
                            onclick="event.stopPropagation(); trackExpedition('${delivery.id}')">
                        <i class="fas fa-map-marker-alt"></i>
                    </button>
                </div>
            </div>
        `;
    });
    
    document.getElementById('deliveriesList').innerHTML = html;
}

function updateTemperatureStats(stats) {
    document.getElementById('tempOkCount').textContent = stats.ok || 0;
    document.getElementById('tempHighCount').textContent = stats.high || 0;
    document.getElementById('tempLowCount').textContent = stats.low || 0;
}

function updateLastUpdate() {
    const now = new Date();
    document.getElementById('lastUpdate').textContent = now.toLocaleTimeString('fr-FR');
}

function refreshMap() {
    loadDeliveries();
}

function centerOnMyLocation() {
    if (navigator.geolocation) {
        navigator.geolocation.getCurrentPosition(position => {
            map.setView([position.coords.latitude, position.coords.longitude], 13);
        });
    } else {
        showAlert('Erreur', 'La géolocalisation n\'est pas supportée par votre navigateur');
    }
}

function filterDeliveries() {
    const filter = document.getElementById('filterStatus').value;
    let filtered = deliveriesData;
    
    if (filter !== 'all') {
        filtered = deliveriesData.filter(d => d.statut === filter);
    }
    
    updateDeliveriesList(filtered);
    updateMapMarkers(filtered);
}

function viewExpedition(expeditionId) {
    window.location.href = `expedition_detail.php?id=${expeditionId}`;
}

function trackExpedition(expeditionId) {
    const delivery = deliveriesData.find(d => d.id === expeditionId);
    if (delivery && delivery.position) {
        map.setView([delivery.position.lat, delivery.position.lon], 13);
        
        // Ouvrir le popup du marqueur correspondant
        markers.forEach(marker => {
            const latlng = marker.getLatLng();
            if (latlng.lat === delivery.position.lat && latlng.lng === delivery.position.lon) {
                marker.openPopup();
            }
        });
    }
}
// Dans votre page map.php ou dashboard.php
document.addEventListener('DOMContentLoaded', function() {
    // Initialiser la carte
    const map = initializeMap('mainMap');
    
    // Charger les livraisons
    API.getActiveDeliveries().then(data => {
        loadDeliveriesOnMap(data.deliveries);
    });
    
    // Centrer sur ma position
    document.getElementById('myLocationBtn').addEventListener('click', () => {
        centerOnMyLocation();
    });
});
</script>

<?php include 'includes/footer.php'; ?>