<?php
session_start();

// Vérifier l'authentification
if (!isset($_SESSION['user'])) {
    header('Location: login.php?error=auth');
    exit();
}

$user = $_SESSION['user'];
$page_title = "Mon profil";
$page_scripts = ['js/profile.js'];

// Traitement de la mise à jour du profil
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    switch ($action) {
        case 'update_profile':
            $name = trim($_POST['name'] ?? '');
            $email = trim($_POST['email'] ?? '');
            $phone = trim($_POST['phone'] ?? '');
            
            // Validation
            if (empty($name) || empty($email)) {
                $error = 'Le nom et l\'email sont obligatoires';
            } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
                $error = 'Format d\'email invalide';
            } else {
                // En production, sauvegarder en base de données
                $_SESSION['user']['name'] = $name;
                $_SESSION['user']['email'] = $email;
                $_SESSION['user']['phone'] = $phone;
                
                $success = 'Profil mis à jour avec succès';
            }
            break;
            
        case 'change_password':
            $current_password = $_POST['current_password'] ?? '';
            $new_password = $_POST['new_password'] ?? '';
            $confirm_password = $_POST['confirm_password'] ?? '';
            
            // Validation
            if (empty($current_password) || empty($new_password) || empty($confirm_password)) {
                $error = 'Tous les champs sont obligatoires';
            } elseif ($new_password !== $confirm_password) {
                $error = 'Les nouveaux mots de passe ne correspondent pas';
            } elseif (strlen($new_password) < 8) {
                $error = 'Le mot de passe doit contenir au moins 8 caractères';
            } else {
                // En production, vérifier le mot de passe actuel en base de données
                // Pour la démo, on suppose que c'est correct
                $success = 'Mot de passe changé avec succès';
            }
            break;
            
        case 'update_notifications':
            $email_notifications = isset($_POST['email_notifications']) ? 1 : 0;
            $push_notifications = isset($_POST['push_notifications']) ? 1 : 0;
            $sms_notifications = isset($_POST['sms_notifications']) ? 1 : 0;
            
            $_SESSION['user']['notifications'] = [
                'email' => $email_notifications,
                'push' => $push_notifications,
                'sms' => $sms_notifications
            ];
            
            $success = 'Préférences de notification mises à jour';
            break;
    }
}

// Récupérer les statistiques de l'utilisateur (simulées)
function getUserStats($user_id, $role) {
    $stats = [
        'transporteur' => [
            'total_deliveries' => 156,
            'on_time_rate' => 95.4,
            'avg_temperature' => 4.2,
            'eco_score' => 92,
            'total_distance' => 12540,
            'carbon_saved' => 1250
        ],
        'producteur' => [
            'total_shipments' => 89,
            'products_shipped' => 24560,
            'avg_quality_score' => 96.7,
            'certified_products' => 100,
            'partners_count' => 12,
            'customer_satisfaction' => 94.2
        ],
        'destinataire' => [
            'total_receptions' => 203,
            'on_time_receptions' => 191,
            'temperature_compliance' => 98.5,
            'quality_issues' => 2.3,
            'verified_shipments' => 100,
            'satisfaction_rate' => 96.8
        ],
        'auditeur' => [
            'audits_completed' => 47,
            'blockchain_verified' => 156,
            'compliance_rate' => 97.3,
            'issues_found' => 12,
            'certifications_granted' => 89,
            'audit_score' => 94.6
        ]
    ];
    
    return $stats[$role] ?? [];
}

$user_stats = getUserStats($user['id'], $user['role']);

include 'includes/header.php';
?>

<div class="page-header">
    <div style="display: flex; align-items: center; gap: 20px;">
        <div style="width: 80px; height: 80px; border-radius: 50%; 
                  background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                  display: flex; align-items: center; justify-content: center;">
            <i class="fas fa-user" style="font-size: 36px; color: white;"></i>
        </div>
        <div>
            <h1><?php echo htmlspecialchars($user['name']); ?></h1>
            <div style="display: flex; align-items: center; gap: 15px; margin-top: 5px;">
                <span class="status-badge" style="background: #3498db; color: white;">
                    <?php 
                    $roles_labels = [
                        'transporteur' => 'Transporteur',
                        'producteur' => 'Producteur',
                        'destinataire' => 'Destinataire',
                        'auditeur' => 'Auditeur'
                    ];
                    echo $roles_labels[$user['role']] ?? 'Utilisateur';
                    ?>
                </span>
                <span style="color: #666;">
                    <i class="fas fa-user-circle"></i> <?php echo htmlspecialchars($user['username']); ?>
                </span>
            </div>
        </div>
    </div>
    
    <div style="display: flex; gap: 10px;">
        <button class="btn btn-success" onclick="showExportModal()">
            <i class="fas fa-file-export"></i> Exporter données
        </button>
        <button class="btn" onclick="showQRCodeModal()">
            <i class="fas fa-qrcode"></i> Mon QR Code
        </button>
    </div>
</div>

<?php if (isset($error)): ?>
    <div class="alert alert-danger fade-in">
        <i class="fas fa-exclamation-triangle"></i> <?php echo htmlspecialchars($error); ?>
    </div>
<?php endif; ?>

<?php if (isset($success)): ?>
    <div class="alert alert-success fade-in">
        <i class="fas fa-check-circle"></i> <?php echo htmlspecialchars($success); ?>
    </div>
<?php endif; ?>

<div class="grid grid-3">
    <!-- Statistiques utilisateur -->
    <div class="card">
        <div class="card-header">
            <h2 class="card-title">
                <i class="fas fa-chart-bar"></i> Mes statistiques
            </h2>
        </div>
        <div style="padding: 20px;">
            <?php if ($user_stats): ?>
                <div style="display: grid; gap: 15px;">
                    <?php foreach ($user_stats as $key => $value): ?>
                        <div style="display: flex; justify-content: space-between; 
                                  padding: 10px 0; border-bottom: 1px solid #eee;">
                            <span style="color: #666;">
                                <?php 
                                $labels = [
                                    'total_deliveries' => 'Livraisons totales',
                                    'on_time_rate' => 'Taux à l\'heure',
                                    'avg_temperature' => 'Température moyenne',
                                    'eco_score' => 'Score écologique',
                                    'total_distance' => 'Distance totale (km)',
                                    'carbon_saved' => 'CO₂ économisé (kg)',
                                    'total_shipments' => 'Expéditions totales',
                                    'products_shipped' => 'Produits expédiés',
                                    'avg_quality_score' => 'Score qualité moyen',
                                    'certified_products' => 'Produits certifiés',
                                    'partners_count' => 'Partenaires',
                                    'customer_satisfaction' => 'Satisfaction clients',
                                    'total_receptions' => 'Réceptions totales',
                                    'on_time_receptions' => 'Réceptions à l\'heure',
                                    'temperature_compliance' => 'Conformité température',
                                    'quality_issues' => 'Problèmes qualité',
                                    'verified_shipments' => 'Expéditions vérifiées',
                                    'satisfaction_rate' => 'Taux satisfaction',
                                    'audits_completed' => 'Audits complétés',
                                    'blockchain_verified' => 'Vérifications blockchain',
                                    'compliance_rate' => 'Taux conformité',
                                    'issues_found' => 'Problèmes identifiés',
                                    'certifications_granted' => 'Certifications accordées',
                                    'audit_score' => 'Score audit'
                                ];
                                echo $labels[$key] ?? $key;
                                ?>
                            </span>
                            <span style="font-weight: 600; color: #2c3e50;">
                                <?php echo is_numeric($value) && $value > 100 ? number_format($value, 0) : $value; ?>
                                <?php echo in_array($key, ['on_time_rate', 'avg_quality_score', 'temperature_compliance', 
                                                         'customer_satisfaction', 'satisfaction_rate', 'compliance_rate', 
                                                         'audit_score', 'eco_score']) ? '%' : ''; ?>
                                <?php echo $key === 'avg_temperature' ? '°C' : ''; ?>
                            </span>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php else: ?>
                <div style="text-align: center; padding: 30px; color: #666;">
                    <i class="fas fa-chart-line" style="font-size: 48px; margin-bottom: 15px;"></i>
                    <p>Aucune statistique disponible</p>
                </div>
            <?php endif; ?>
        </div>
    </div>
    
    <!-- Informations du compte -->
    <div class="card">
        <div class="card-header">
            <h2 class="card-title">
                <i class="fas fa-user-cog"></i> Informations du compte
            </h2>
        </div>
        <form method="POST" action="">
            <input type="hidden" name="action" value="update_profile">
            
            <div style="padding: 20px;">
                <div class="form-group">
                    <label for="name">Nom complet *</label>
                    <input type="text" id="name" name="name" required 
                           value="<?php echo htmlspecialchars($user['name']); ?>">
                </div>
                
                <div class="form-group">
                    <label for="email">Email *</label>
                    <input type="email" id="email" name="email" required 
                           value="<?php echo htmlspecialchars($user['email']); ?>">
                </div>
                
                <div class="form-group">
                    <label for="phone">Téléphone</label>
                    <input type="tel" id="phone" name="phone" 
                           value="<?php echo htmlspecialchars($user['phone'] ?? ''); ?>">
                </div>
                
                <div class="form-group">
                    <label for="username">Nom d'utilisateur</label>
                    <input type="text" id="username" value="<?php echo htmlspecialchars($user['username']); ?>" disabled>
                    <div style="font-size: 12px; color: #666; margin-top: 5px;">
                        Le nom d'utilisateur ne peut pas être modifié
                    </div>
                </div>
                
                <div class="form-group">
                    <label>Rôle</label>
                    <input type="text" value="<?php echo htmlspecialchars(ucfirst($user['role'])); ?>" disabled>
                </div>
                
                <div class="form-group">
                    <label>Membre depuis</label>
                    <input type="text" value="<?php echo date('d/m/Y', $user['login_time'] ?? time()); ?>" disabled>
                </div>
            </div>
            
            <div class="modal-footer" style="border-top: 1px solid #eee; padding: 20px;">
                <button type="submit" class="btn btn-success">
                    <i class="fas fa-save"></i> Enregistrer les modifications
                </button>
            </div>
        </form>
    </div>
    
    <!-- Sécurité et notifications -->
    <div class="card">
        <div class="card-header">
            <h2 class="card-title">
                <i class="fas fa-shield-alt"></i> Sécurité
            </h2>
        </div>
        
        <!-- Changement de mot de passe -->
        <div style="padding: 20px; border-bottom: 1px solid #eee;">
            <h3 style="margin-bottom: 15px; color: #2c3e50; font-size: 16px;">
                <i class="fas fa-key"></i> Changer le mot de passe
            </h3>
            <form method="POST" action="" id="passwordForm">
                <input type="hidden" name="action" value="change_password">
                
                <div class="form-group">
                    <label for="current_password">Mot de passe actuel *</label>
                    <div style="position: relative;">
                        <input type="password" id="current_password" name="current_password" required>
                        <button type="button" class="toggle-password" data-target="current_password" 
                                style="position: absolute; right: 10px; top: 50%; transform: translateY(-50%);
                                       background: none; border: none; color: #666; cursor: pointer;">
                            <i class="fas fa-eye"></i>
                        </button>
                    </div>
                </div>
                
                <div class="form-group">
                    <label for="new_password">Nouveau mot de passe *</label>
                    <div style="position: relative;">
                        <input type="password" id="new_password" name="new_password" required 
                               pattern=".{8,}" title="8 caractères minimum">
                        <button type="button" class="toggle-password" data-target="new_password"
                                style="position: absolute; right: 10px; top: 50%; transform: translateY(-50%);
                                       background: none; border: none; color: #666; cursor: pointer;">
                            <i class="fas fa-eye"></i>
                        </button>
                    </div>
                    <div style="font-size: 12px; color: #666; margin-top: 5px;">
                        Minimum 8 caractères
                    </div>
                </div>
                
                <div class="form-group">
                    <label for="confirm_password">Confirmer le mot de passe *</label>
                    <div style="position: relative;">
                        <input type="password" id="confirm_password" name="confirm_password" required>
                        <button type="button" class="toggle-password" data-target="confirm_password"
                                style="position: absolute; right: 10px; top: 50%; transform: translateY(-50%);
                                       background: none; border: none; color: #666; cursor: pointer;">
                            <i class="fas fa-eye"></i>
                        </button>
                    </div>
                </div>
                
                <button type="submit" class="btn btn-block" style="margin-top: 10px;">
                    <i class="fas fa-sync-alt"></i> Changer le mot de passe
                </button>
            </form>
        </div>
        
        <!-- Authentification à deux facteurs -->
        <div style="padding: 20px; border-bottom: 1px solid #eee;">
            <h3 style="margin-bottom: 15px; color: #2c3e50; font-size: 16px;">
                <i class="fas fa-mobile-alt"></i> Authentification à deux facteurs (2FA)
            </h3>
            
            <div style="display: flex; justify-content: space-between; align-items: center;">
                <div>
                    <div style="font-weight: 600;">Statut 2FA</div>
                    <div style="font-size: 14px; color: #666;">
                        <?php echo ($user['2fa_enabled'] ?? false) ? 'Activé' : 'Désactivé'; ?>
                    </div>
                </div>
                <button class="btn <?php echo ($user['2fa_enabled'] ?? false) ? 'btn-danger' : 'btn-success'; ?>" 
                        onclick="toggle2FA()" style="padding: 8px 16px;">
                    <i class="fas <?php echo ($user['2fa_enabled'] ?? false) ? 'fa-lock-open' : 'fa-lock'; ?>"></i>
                    <?php echo ($user['2fa_enabled'] ?? false) ? 'Désactiver 2FA' : 'Activer 2FA'; ?>
                </button>
            </div>
            
            <div style="margin-top: 15px; font-size: 14px; color: #666;">
                <i class="fas fa-info-circle"></i> Ajoutez une couche de sécurité supplémentaire avec l'authentification à deux facteurs.
            </div>
        </div>
        
        <!-- Notifications -->
        <div style="padding: 20px;">
            <h3 style="margin-bottom: 15px; color: #2c3e50; font-size: 16px;">
                <i class="fas fa-bell"></i> Préférences de notification
            </h3>
            <form method="POST" action="" id="notificationsForm">
                <input type="hidden" name="action" value="update_notifications">
                
                <?php 
                $notifications = $user['notifications'] ?? [
                    'email' => true,
                    'push' => true,
                    'sms' => false
                ];
                ?>
                
                <div style="display: grid; gap: 15px;">
                    <label style="display: flex; align-items: center; gap: 10px; cursor: pointer;">
                        <input type="checkbox" name="email_notifications" 
                               <?php echo $notifications['email'] ? 'checked' : ''; ?>>
                        <div>
                            <div style="font-weight: 600;">Notifications par email</div>
                            <div style="font-size: 12px; color: #666;">
                                Alertes, rapports et mises à jour importantes
                            </div>
                        </div>
                    </label>
                    
                    <label style="display: flex; align-items: center; gap: 10px; cursor: pointer;">
                        <input type="checkbox" name="push_notifications" 
                               <?php echo $notifications['push'] ? 'checked' : ''; ?>>
                        <div>
                            <div style="font-weight: 600;">Notifications push</div>
                            <div style="font-size: 12px; color: #666;">
                                Alertes en temps réel sur le site
                            </div>
                        </div>
                    </label>
                    
                    <label style="display: flex; align-items: center; gap: 10px; cursor: pointer;">
                        <input type="checkbox" name="sms_notifications" 
                               <?php echo $notifications['sms'] ? 'checked' : ''; ?>>
                        <div>
                            <div style="font-weight: 600;">Notifications SMS</div>
                            <div style="font-size: 12px; color: #666;">
                                Alertes urgentes par SMS
                            </div>
                        </div>
                    </label>
                </div>
                
                <button type="submit" class="btn btn-block" style="margin-top: 20px;">
                    <i class="fas fa-save"></i> Enregistrer les préférences
                </button>
            </form>
        </div>
    </div>
</div>

<!-- Sessions actives -->
<div class="card">
    <div class="card-header">
        <h2 class="card-title">
            <i class="fas fa-desktop"></i> Sessions actives
        </h2>
        <button class="btn btn-danger" onclick="logoutAllSessions()" style="padding: 8px 16px;">
            <i class="fas fa-sign-out-alt"></i> Déconnecter toutes les sessions
        </button>
    </div>
    <div class="table-responsive">
        <table class="data-table">
            <thead>
                <tr>
                    <th>Appareil</th>
                    <th>Navigateur</th>
                    <th>IP</th>
                    <th>Dernière activité</th>
                    <th>Localisation</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
                <tr>
                    <td>
                        <div style="display: flex; align-items: center; gap: 10px;">
                            <i class="fas fa-desktop"></i>
                            <span>PC Bureau - Windows</span>
                        </div>
                    </td>
                    <td>Chrome 120</td>
                    <td>192.168.1.100</td>
                    <td>À l'instant</td>
                    <td>Paris, France</td>
                    <td>
                        <button class="btn btn-danger" style="padding: 5px 10px; font-size: 12px;">
                            <i class="fas fa-times"></i> Déconnecter
                        </button>
                    </td>
                </tr>
                <tr>
                    <td>
                        <div style="display: flex; align-items: center; gap: 10px;">
                            <i class="fas fa-mobile-alt"></i>
                            <span>iPhone 13</span>
                        </div>
                    </td>
                    <td>Safari 16</td>
                    <td>192.168.1.101</td>
                    <td>Il y a 2 heures</td>
                    <td>Paris, France</td>
                    <td>
                        <button class="btn btn-danger" style="padding: 5px 10px; font-size: 12px;">
                            <i class="fas fa-times"></i> Déconnecter
                        </button>
                    </td>
                </tr>
            </tbody>
        </table>
    </div>
</div>

<!-- Permissions et accès -->
<div class="card">
    <div class="card-header">
        <h2 class="card-title">
            <i class="fas fa-user-shield"></i> Permissions et accès
        </h2>
    </div>
    <div style="padding: 20px;">
        <div style="display: grid; grid-template-columns: repeat(auto-fill, minmax(250px, 1fr)); gap: 15px;">
            <?php foreach ($user['permissions'] as $permission): ?>
                <div style="background: #f8f9fa; padding: 15px; border-radius: 8px; display: flex; align-items: center; gap: 10px;">
                    <div style="width: 30px; height: 30px; border-radius: 50%; background: #27ae60; 
                              display: flex; align-items: center; justify-content: center; color: white;">
                        <i class="fas fa-check"></i>
                    </div>
                    <div>
                        <div style="font-weight: 600;"><?php echo getPermissionLabel($permission); ?></div>
                        <div style="font-size: 12px; color: #666;">Accès autorisé</div>
                    </div>
                </div>
            <?php endforeach; ?>
        </div>
        
        <div style="margin-top: 30px;">
            <h3 style="margin-bottom: 15px; color: #2c3e50;">Demander des accès supplémentaires</h3>
            <form onsubmit="requestAccess(event)" style="display: flex; gap: 10px;">
                <select style="flex: 1; padding: 10px; border: 1px solid #ddd; border-radius: 4px;">
                    <option value="">Sélectionner un accès...</option>
                    <option value="blockchain:write">Écriture blockchain</option>
                    <option value="audit:write">Écriture audit</option>
                    <option value="admin:view">Vue administrateur</option>
                    <option value="export:full">Export complet</option>
                </select>
                <button type="submit" class="btn" style="white-space: nowrap;">
                    <i class="fas fa-paper-plane"></i> Envoyer la demande
                </button>
            </form>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Toggle password visibility
    document.querySelectorAll('.toggle-password').forEach(button => {
        button.addEventListener('click', function() {
            const targetId = this.getAttribute('data-target');
            const input = document.getElementById(targetId);
            if (input) {
                const type = input.getAttribute('type') === 'password' ? 'text' : 'password';
                input.setAttribute('type', type);
                
                const icon = this.querySelector('i');
                if (type === 'text') {
                    icon.className = 'fas fa-eye-slash';
                } else {
                    icon.className = 'fas fa-eye';
                }
            }
        });
    });
    
    // Validation du formulaire de mot de passe
    const passwordForm = document.getElementById('passwordForm');
    if (passwordForm) {
        passwordForm.addEventListener('submit', function(e) {
            const newPassword = document.getElementById('new_password').value;
            const confirmPassword = document.getElementById('confirm_password').value;
            
            if (newPassword !== confirmPassword) {
                e.preventDefault();
                showAlert('Erreur', 'Les mots de passe ne correspondent pas');
                return false;
            }
            
            if (newPassword.length < 8) {
                e.preventDefault();
                showAlert('Erreur', 'Le mot de passe doit contenir au moins 8 caractères');
                return false;
            }
            
            return true;
        });
    }
});

function toggle2FA() {
    showModal('2faModal', {
        title: 'Authentification à deux facteurs',
        content: `
            <div style="text-align: center; padding: 20px;">
                <div style="width: 200px; height: 200px; background: #f8f9fa; border-radius: 8px;
                          display: flex; align-items: center; justify-content: center; margin: 0 auto 20px;">
                    <div style="font-size: 48px; color: #3498db;">
                        <i class="fas fa-qrcode"></i>
                    </div>
                </div>
                <div style="margin-bottom: 20px;">
                    <p>Scannez ce QR Code avec votre application d'authentification (Google Authenticator, Authy, etc.)</p>
                    <p style="font-family: monospace; background: #f8f9fa; padding: 10px; border-radius: 4px; margin: 10px 0;">
                        <?php echo strtoupper(bin2hex(random_bytes(10))); ?>
                    </p>
                </div>
                <div class="form-group">
                    <label>Code de vérification</label>
                    <input type="text" id="2faCode" placeholder="000000" maxlength="6" 
                           style="text-align: center; letter-spacing: 10px; font-size: 24px;">
                </div>
            </div>
        `,
        buttons: `
            <button class="btn" onclick="closeModal('2faModal')">Annuler</button>
            <button class="btn btn-success" onclick="enable2FA()">Activer 2FA</button>
        `
    });
}

function enable2FA() {
    const code = document.getElementById('2faCode').value;
    if (code.length !== 6) {
        showAlert('Erreur', 'Veuillez entrer un code à 6 chiffres');
        return;
    }
    
    // Simuler l'activation 2FA
    showAlert('Succès', 'Authentification à deux facteurs activée avec succès', 'success');
    closeModal('2faModal');
    
    // Recharger la page après 2 secondes
    setTimeout(() => {
        window.location.reload();
    }, 2000);
}

function logoutAllSessions() {
    if (confirm('Êtes-vous sûr de vouloir déconnecter toutes les sessions ?')) {
        // En production, appeler l'API pour déconnecter toutes les sessions
        showAlert('Information', 'Toutes les sessions ont été déconnectées', 'info');
        
        setTimeout(() => {
            window.location.href = 'api/logout.php?all=true';
        }, 1500);
    }
}

function requestAccess(e) {
    e.preventDefault();
    const select = e.target.querySelector('select');
    const access = select.value;
    
    if (!access) {
        showAlert('Erreur', 'Veuillez sélectionner un type d\'accès');
        return;
    }
    
    // Simuler l'envoi de la demande
    showAlert('Succès', 'Votre demande d\'accès a été envoyée à l\'administrateur', 'success');
    select.value = '';
}

function showExportModal() {
    showModal('exportModal', {
        title: 'Exporter mes données',
        content: `
            <div style="padding: 20px;">
                <p>Sélectionnez les données que vous souhaitez exporter :</p>
                <div style="display: grid; gap: 15px; margin: 20px 0;">
                    <label style="display: flex; align-items: center; gap: 10px; cursor: pointer;">
                        <input type="checkbox" checked>
                        <span>Informations personnelles</span>
                    </label>
                    <label style="display: flex; align-items: center; gap: 10px; cursor: pointer;">
                        <input type="checkbox" checked>
                        <span>Historique des expéditions</span>
                    </label>
                    <label style="display: flex; align-items: center; gap: 10px; cursor: pointer;">
                        <input type="checkbox">
                        <span>Données de traçabilité</span>
                    </label>
                    <label style="display: flex; align-items: center; gap: 10px; cursor: pointer;">
                        <input type="checkbox">
                        <span>Preuves blockchain</span>
                    </label>
                    <label style="display: flex; align-items: center; gap: 10px; cursor: pointer;">
                        <input type="checkbox" checked>
                        <span>Statistiques et rapports</span>
                    </label>
                </div>
                <div class="form-group">
                    <label>Format d'export</label>
                    <select id="exportFormat">
                        <option value="json">JSON</option>
                        <option value="csv">CSV</option>
                        <option value="pdf">PDF</option>
                        <option value="excel">Excel</option>
                    </select>
                </div>
                <div class="form-group">
                    <label>Période</label>
                    <div style="display: flex; gap: 10px;">
                        <input type="date" id="exportStart" style="flex: 1;">
                        <span>à</span>
                        <input type="date" id="exportEnd" style="flex: 1;">
                    </div>
                </div>
            </div>
        `,
        buttons: `
            <button class="btn" onclick="closeModal('exportModal')">Annuler</button>
            <button class="btn btn-success" onclick="exportData()">
                <i class="fas fa-download"></i> Exporter
            </button>
        `
    });
    
    // Définir les dates par défaut
    const today = new Date().toISOString().split('T')[0];
    const oneMonthAgo = new Date();
    oneMonthAgo.setMonth(oneMonthAgo.getMonth() - 1);
    const oneMonthAgoStr = oneMonthAgo.toISOString().split('T')[0];
    
    document.getElementById('exportStart').value = oneMonthAgoStr;
    document.getElementById('exportEnd').value = today;
}

function exportData() {
    const format = document.getElementById('exportFormat').value;
    
    // Simuler l'export
    showAlert('Information', `Export en cours au format ${format.toUpperCase()}...`, 'info');
    closeModal('exportModal');
    
    // Simuler le téléchargement
    setTimeout(() => {
        showAlert('Succès', 'Export terminé. Votre fichier est prêt au téléchargement.', 'success');
    }, 2000);
}

function showQRCodeModal() {
    const userData = <?php echo json_encode($user); ?>;
    
    showModal('qrCodeModal', {
        title: 'Mon QR Code d\'identification',
        content: `
            <div style="text-align: center; padding: 20px;">
                <div style="width: 250px; height: 250px; background: #f8f9fa; border-radius: 8px;
                          display: flex; align-items: center; justify-content: center; margin: 0 auto 20px;">
                    <div style="font-size: 64px; color: #3498db;">
                        <i class="fas fa-qrcode"></i>
                    </div>
                </div>
                <div style="margin-bottom: 20px;">
                    <p>Utilisez ce QR Code pour vous identifier rapidement</p>
                    <div style="background: #f8f9fa; padding: 15px; border-radius: 8px; margin: 15px 0; font-size: 12px;">
                        <div><strong>ID:</strong> ${userData.id}</div>
                        <div><strong>Nom:</strong> ${userData.name}</div>
                        <div><strong>Rôle:</strong> ${userData.role}</div>
                    </div>
                </div>
                <div style="display: flex; gap: 10px; justify-content: center;">
                    <button class="btn" onclick="downloadQRCode()">
                        <i class="fas fa-download"></i> Télécharger
                    </button>
                    <button class="btn" onclick="printQRCode()">
                        <i class="fas fa-print"></i> Imprimer
                    </button>
                </div>
            </div>
        `,
        buttons: `
            <button class="btn" onclick="closeModal('qrCodeModal')">Fermer</button>
        `
    });
}

function downloadQRCode() {
    showAlert('Information', 'Téléchargement du QR Code...', 'info');
    // Ici, générer et télécharger le vrai QR Code
}

function printQRCode() {
    window.print();
}
</script>

<?php
// Fonction pour obtenir le libellé d'une permission
function getPermissionLabel($permission) {
    $labels = [
        'expedition:read' => 'Lecture des expéditions',
        'expedition:create' => 'Création d\'expéditions',
        'expedition:delete' => 'Suppression d\'expéditions',
        'scan:write' => 'Écriture des scans',
        'iot:read' => 'Lecture données IoT',
        'map:view' => 'Visualisation carte',
        'delivery:validate' => 'Validation livraison',
        'blockchain:view' => 'Visualisation blockchain',
        'audit:read' => 'Lecture audits',
        'report:view' => 'Visualisation rapports',
        'profile:edit' => 'Modification profil'
    ];
    
    return $labels[$permission] ?? $permission;
}
?>

<?php include 'includes/footer.php'; ?>