<?php
session_start();

// Si déjà connecté, rediriger vers le dashboard
if (isset($_SESSION['user'])) {
    header('Location: dashboard.php');
    exit();
}

// Traitement du formulaire d'inscription
$errors = [];
$success = false;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Récupérer et valider les données
    $name = trim($_POST['name'] ?? '');
    $email = trim($_POST['email'] ?? '');
    $username = trim($_POST['username'] ?? '');
    $password = $_POST['password'] ?? '';
    $confirm_password = $_POST['confirm_password'] ?? '';
    $role = $_POST['role'] ?? '';
    $company = trim($_POST['company'] ?? '');
    $phone = trim($_POST['phone'] ?? '');
    $terms = isset($_POST['terms']);
    
    // Validation
    if (empty($name)) {
        $errors['name'] = 'Le nom complet est obligatoire';
    } elseif (strlen($name) < 2) {
        $errors['name'] = 'Le nom doit contenir au moins 2 caractères';
    }
    
    if (empty($email)) {
        $errors['email'] = 'L\'email est obligatoire';
    } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $errors['email'] = 'Format d\'email invalide';
    }
    
    if (empty($username)) {
        $errors['username'] = 'Le nom d\'utilisateur est obligatoire';
    } elseif (strlen($username) < 3) {
        $errors['username'] = 'Le nom d\'utilisateur doit contenir au moins 3 caractères';
    } elseif (!preg_match('/^[a-zA-Z0-9_]+$/', $username)) {
        $errors['username'] = 'Caractères autorisés : lettres, chiffres et underscore';
    }
    
    if (empty($password)) {
        $errors['password'] = 'Le mot de passe est obligatoire';
    } elseif (strlen($password) < 8) {
        $errors['password'] = 'Le mot de passe doit contenir au moins 8 caractères';
    } elseif (!preg_match('/[A-Z]/', $password) || !preg_match('/[a-z]/', $password) || !preg_match('/[0-9]/', $password)) {
        $errors['password'] = 'Le mot de passe doit contenir au moins une majuscule, une minuscule et un chiffre';
    }
    
    if ($password !== $confirm_password) {
        $errors['confirm_password'] = 'Les mots de passe ne correspondent pas';
    }
    
    if (empty($role)) {
        $errors['role'] = 'Le rôle est obligatoire';
    }
    
    if (empty($company)) {
        $errors['company'] = 'Le nom de l\'entreprise est obligatoire';
    }
    
    if (!$terms) {
        $errors['terms'] = 'Vous devez accepter les conditions d\'utilisation';
    }
    
    // Vérifier si l'utilisateur existe déjà (simulation)
    $existingUsers = [
        'transporteur_1',
        'producteur_1',
        'destinataire_1',
        'auditeur_1'
    ];
    
    if (in_array($username, $existingUsers)) {
        $errors['username'] = 'Ce nom d\'utilisateur est déjà pris';
    }
    
    // Si aucune erreur, créer le compte
    if (empty($errors)) {
        // En production, vous enregistreriez dans la base de données
        // Pour la démo, on simule la création
        
        // Générer un token de vérification email
        $verification_token = bin2hex(random_bytes(32));
        
        // Préparer les données utilisateur
        $user_data = [
            'id' => 'user_' . uniqid(),
            'name' => $name,
            'email' => $email,
            'username' => $username,
            'password_hash' => password_hash($password, PASSWORD_DEFAULT),
            'role' => $role,
            'company' => $company,
            'phone' => $phone,
            'verification_token' => $verification_token,
            'verified' => false,
            'created_at' => time(),
            'status' => 'pending', // pending, active, suspended
            'permissions' => getDefaultPermissions($role)
        ];
        
        // Envoyer un email de vérification (simulé)
        sendVerificationEmail($email, $name, $verification_token);
        
        // Stocker temporairement les données (en production, base de données)
        $_SESSION['registration_pending'] = [
            'user_data' => $user_data,
            'expires' => time() + 3600 // 1 heure
        ];
        
        // Rediriger vers la page de confirmation
        header('Location: register_confirmation.php?email=' . urlencode($email));
        exit();
    }
}

// Fonction pour obtenir les permissions par défaut
function getDefaultPermissions($role) {
    $permissions = [
        'transporteur' => [
            'expedition:read',
            'expedition:create',
            'scan:write',
            'iot:read',
            'map:view',
            'profile:edit'
        ],
        'producteur' => [
            'expedition:read',
            'expedition:create',
            'expedition:delete',
            'report:view',
            'profile:edit'
        ],
        'destinataire' => [
            'expedition:read',
            'delivery:validate',
            'report:view',
            'profile:edit'
        ],
        'auditeur' => [
            'expedition:read',
            'blockchain:view',
            'audit:read',
            'report:view',
            'profile:edit'
        ]
    ];
    
    return $permissions[$role] ?? [];
}

// Fonction pour envoyer un email de vérification (simulée)
function sendVerificationEmail($email, $name, $token) {
    // En production, utiliser PHPMailer, SwiftMailer ou un service d'email
    $verification_link = "http://" . $_SERVER['HTTP_HOST'] . dirname($_SERVER['PHP_SELF']) . "/verify_email.php?token=" . $token;
    
    // Pour la démo, on log seulement
    error_log("Email de vérification envoyé à $email");
    error_log("Lien de vérification : $verification_link");
    
    return true;
}
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Blockchain Logistique - Inscription</title>
    <link rel="stylesheet" href="css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        /* Styles spécifiques pour la page d'inscription */
        .register-container {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }
        
        .register-box {
            background: white;
            border-radius: 20px;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
            overflow: hidden;
            max-width: 1000px;
            width: 100%;
            min-height: 600px;
        }
        
        .register-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 40px;
            text-align: center;
        }
        
        .register-header h1 {
            font-size: 36px;
            margin-bottom: 10px;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 15px;
        }
        
        .register-header p {
            font-size: 18px;
            opacity: 0.9;
            max-width: 600px;
            margin: 0 auto;
        }
        
        .register-content {
            padding: 40px;
            max-height: 800px;
            overflow-y: auto;
        }
        
        .register-steps {
            display: flex;
            justify-content: space-between;
            margin-bottom: 40px;
            position: relative;
        }
        
        .register-steps:before {
            content: '';
            position: absolute;
            top: 25px;
            left: 0;
            right: 0;
            height: 2px;
            background: #eee;
            z-index: 1;
        }
        
        .step {
            position: relative;
            z-index: 2;
            text-align: center;
            flex: 1;
        }
        
        .step-number {
            width: 50px;
            height: 50px;
            border-radius: 50%;
            background: white;
            border: 2px solid #eee;
            display: flex;
            align-items: center;
            justify-content: center;
            margin: 0 auto 10px;
            font-weight: 600;
            font-size: 18px;
            color: #999;
        }
        
        .step.active .step-number {
            background: #3498db;
            border-color: #3498db;
            color: white;
        }
        
        .step.completed .step-number {
            background: #27ae60;
            border-color: #27ae60;
            color: white;
        }
        
        .step-label {
            font-size: 14px;
            color: #666;
            font-weight: 500;
        }
        
        .step.active .step-label {
            color: #3498db;
            font-weight: 600;
        }
        
        .register-section {
            display: none;
        }
        
        .register-section.active {
            display: block;
            animation: fadeIn 0.5s ease;
        }
        
        .form-grid {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 20px;
        }
        
        .form-group.has-error input,
        .form-group.has-error select,
        .form-group.has-error textarea {
            border-color: #e74c3c;
        }
        
        .error-message {
            color: #e74c3c;
            font-size: 12px;
            margin-top: 5px;
            display: flex;
            align-items: center;
            gap: 5px;
        }
        
        .password-strength {
            margin-top: 5px;
            height: 5px;
            background: #eee;
            border-radius: 3px;
            overflow: hidden;
        }
        
        .password-strength-bar {
            height: 100%;
            width: 0%;
            transition: width 0.3s ease;
        }
        
        .password-strength.weak .password-strength-bar {
            background: #e74c3c;
            width: 33%;
        }
        
        .password-strength.medium .password-strength-bar {
            background: #f39c12;
            width: 66%;
        }
        
        .password-strength.strong .password-strength-bar {
            background: #27ae60;
            width: 100%;
        }
        
        .password-requirements {
            margin-top: 10px;
            font-size: 12px;
            color: #666;
        }
        
        .requirement {
            display: flex;
            align-items: center;
            gap: 5px;
            margin-bottom: 3px;
        }
        
        .requirement.met {
            color: #27ae60;
        }
        
        .requirement.unmet {
            color: #666;
        }
        
        .requirement i {
            font-size: 10px;
        }
        
        .register-footer {
            display: flex;
            justify-content: space-between;
            margin-top: 40px;
            padding-top: 20px;
            border-top: 1px solid #eee;
        }
        
        .role-card {
            border: 2px solid #eee;
            border-radius: 10px;
            padding: 20px;
            cursor: pointer;
            transition: all 0.3s ease;
            text-align: center;
        }
        
        .role-card:hover {
            border-color: #3498db;
            transform: translateY(-5px);
        }
        
        .role-card.selected {
            border-color: #3498db;
            background: #f8fafd;
        }
        
        .role-card i {
            font-size: 36px;
            color: #3498db;
            margin-bottom: 15px;
        }
        
        .role-card h3 {
            margin-bottom: 10px;
            color: #2c3e50;
        }
        
        .role-card p {
            font-size: 14px;
            color: #666;
            margin-bottom: 10px;
        }
        
        .role-features {
            font-size: 12px;
            color: #666;
            text-align: left;
            margin-top: 15px;
        }
        
        .role-features ul {
            list-style: none;
            padding: 0;
            margin: 0;
        }
        
        .role-features li {
            margin-bottom: 5px;
            display: flex;
            align-items: center;
            gap: 5px;
        }
        
        .role-features i {
            font-size: 12px;
            color: #27ae60;
        }
        
        @media (max-width: 768px) {
            .form-grid {
                grid-template-columns: 1fr;
            }
            
            .register-steps {
                flex-wrap: wrap;
                gap: 20px;
            }
            
            .step {
                flex: none;
                width: 100px;
            }
        }
        
        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(20px); }
            to { opacity: 1; transform: translateY(0); }
        }
    </style>
</head>
<body>
    <div class="register-container">
        <div class="register-box fade-in">
            <!-- En-tête -->
            <div class="register-header">
                <h1>
                    <i class="fas fa-user-plus"></i>
                    Créer un compte
                </h1>
                <p>Rejoignez notre plateforme de logistique blockchain et bénéficiez d'une traçabilité complète et sécurisée</p>
            </div>
            
            <!-- Étapes de l'inscription -->
            <div class="register-content">
                <div class="register-steps">
                    <div class="step active" id="step1">
                        <div class="step-number">1</div>
                        <div class="step-label">Informations</div>
                    </div>
                    <div class="step" id="step2">
                        <div class="step-number">2</div>
                        <div class="step-label">Rôle</div>
                    </div>
                    <div class="step" id="step3">
                        <div class="step-number">3</div>
                        <div class="step-label">Sécurité</div>
                    </div>
                    <div class="step" id="step4">
                        <div class="step-number">4</div>
                        <div class="step-label">Confirmation</div>
                    </div>
                </div>
                
                <!-- Formulaire d'inscription -->
                <form method="POST" action="" id="registerForm">
                    <!-- Étape 1 : Informations personnelles -->
                    <div class="register-section active" id="section1">
                        <h2 style="color: #2c3e50; margin-bottom: 30px;">
                            <i class="fas fa-user"></i> Informations personnelles
                        </h2>
                        
                        <div class="form-grid">
                            <div class="form-group <?php echo isset($errors['name']) ? 'has-error' : ''; ?>">
                                <label for="name">Nom complet *</label>
                                <input type="text" 
                                       id="name" 
                                       name="name" 
                                       required
                                       value="<?php echo htmlspecialchars($_POST['name'] ?? ''); ?>"
                                       placeholder="Jean Dupont">
                                <?php if (isset($errors['name'])): ?>
                                    <div class="error-message">
                                        <i class="fas fa-exclamation-circle"></i>
                                        <?php echo htmlspecialchars($errors['name']); ?>
                                    </div>
                                <?php endif; ?>
                            </div>
                            
                            <div class="form-group <?php echo isset($errors['email']) ? 'has-error' : ''; ?>">
                                <label for="email">Adresse email *</label>
                                <input type="email" 
                                       id="email" 
                                       name="email" 
                                       required
                                       value="<?php echo htmlspecialchars($_POST['email'] ?? ''); ?>"
                                       placeholder="jean.dupont@entreprise.fr">
                                <?php if (isset($errors['email'])): ?>
                                    <div class="error-message">
                                        <i class="fas fa-exclamation-circle"></i>
                                        <?php echo htmlspecialchars($errors['email']); ?>
                                    </div>
                                <?php endif; ?>
                            </div>
                            
                            <div class="form-group <?php echo isset($errors['company']) ? 'has-error' : ''; ?>">
                                <label for="company">Entreprise *</label>
                                <input type="text" 
                                       id="company" 
                                       name="company" 
                                       required
                                       value="<?php echo htmlspecialchars($_POST['company'] ?? ''); ?>"
                                       placeholder="Nom de votre entreprise">
                                <?php if (isset($errors['company'])): ?>
                                    <div class="error-message">
                                        <i class="fas fa-exclamation-circle"></i>
                                        <?php echo htmlspecialchars($errors['company']); ?>
                                    </div>
                                <?php endif; ?>
                            </div>
                            
                            <div class="form-group">
                                <label for="phone">Téléphone</label>
                                <input type="tel" 
                                       id="phone" 
                                       name="phone"
                                       value="<?php echo htmlspecialchars($_POST['phone'] ?? ''); ?>"
                                       placeholder="+33 1 23 45 67 89">
                            </div>
                        </div>
                        
                        <div style="margin-top: 30px;">
                            <button type="button" class="btn btn-next" onclick="nextStep(2)" 
                                    style="float: right; background: #3498db;">
                                <i class="fas fa-arrow-right"></i> Suivant
                            </button>
                        </div>
                    </div>
                    
                    <!-- Étape 2 : Rôle professionnel -->
                    <div class="register-section" id="section2">
                        <h2 style="color: #2c3e50; margin-bottom: 30px;">
                            <i class="fas fa-briefcase"></i> Rôle professionnel
                        </h2>
                        
                        <div style="color: #666; margin-bottom: 30px;">
                            <p>Sélectionnez votre rôle principal sur la plateforme. Cette information déterminera vos permissions et accès.</p>
                        </div>
                        
                        <div class="form-grid" style="margin-bottom: 30px;">
                            <div class="role-card" onclick="selectRole('transporteur')">
                                <i class="fas fa-truck"></i>
                                <h3>Transporteur</h3>
                                <p>Vous gérez le transport et la livraison des produits</p>
                                <div class="role-features">
                                    <ul>
                                        <li><i class="fas fa-check"></i> Suivi en temps réel</li>
                                        <li><i class="fas fa-check"></i> Scan QR Code</li>
                                        <li><i class="fas fa-check"></i> Gestion des expéditions</li>
                                    </ul>
                                </div>
                                <input type="radio" name="role" value="transporteur" style="display: none;">
                            </div>
                            
                            <div class="role-card" onclick="selectRole('producteur')">
                                <i class="fas fa-seedling"></i>
                                <h3>Producteur</h3>
                                <p>Vous produisez et expédiez des marchandises</p>
                                <div class="role-features">
                                    <ul>
                                        <li><i class="fas fa-check"></i> Création d'expéditions</li>
                                        <li><i class="fas fa-check"></i> Traçabilité produit</li>
                                        <li><i class="fas fa-check"></i> Certifications</li>
                                    </ul>
                                </div>
                                <input type="radio" name="role" value="producteur" style="display: none;">
                            </div>
                            
                            <div class="role-card" onclick="selectRole('destinataire')">
                                <i class="fas fa-warehouse"></i>
                                <h3>Destinataire</h3>
                                <p>Vous recevez et validez les livraisons</p>
                                <div class="role-features">
                                    <ul>
                                        <li><i class="fas fa-check"></i> Validation de livraison</li>
                                        <li><i class="fas fa-check"></i> Suivi des réceptions</li>
                                        <li><i class="fas fa-check"></i> Contrôle qualité</li>
                                    </ul>
                                </div>
                                <input type="radio" name="role" value="destinataire" style="display: none;">
                            </div>
                            
                            <div class="role-card" onclick="selectRole('auditeur')">
                                <i class="fas fa-clipboard-check"></i>
                                <h3>Auditeur</h3>
                                <p>Vous vérifiez et auditez la chaîne logistique</p>
                                <div class="role-features">
                                    <ul>
                                        <li><i class="fas fa-check"></i> Audit blockchain</li>
                                        <li><i class="fas fa-check"></i> Vérification des preuves</li>
                                        <li><i class="fas fa-check"></i> Rapports de conformité</li>
                                    </ul>
                                </div>
                                <input type="radio" name="role" value="auditeur" style="display: none;">
                            </div>
                        </div>
                        
                        <?php if (isset($errors['role'])): ?>
                            <div class="error-message" style="margin-bottom: 20px;">
                                <i class="fas fa-exclamation-circle"></i>
                                <?php echo htmlspecialchars($errors['role']); ?>
                            </div>
                        <?php endif; ?>
                        
                        <div style="margin-top: 30px;">
                            <button type="button" class="btn" onclick="prevStep(1)" 
                                    style="background: #95a5a6;">
                                <i class="fas fa-arrow-left"></i> Retour
                            </button>
                            <button type="button" class="btn btn-next" onclick="nextStep(3)" 
                                    style="float: right; background: #3498db;">
                                <i class="fas fa-arrow-right"></i> Suivant
                            </button>
                        </div>
                    </div>
                    
                    <!-- Étape 3 : Sécurité du compte -->
                    <div class="register-section" id="section3">
                        <h2 style="color: #2c3e50; margin-bottom: 30px;">
                            <i class="fas fa-shield-alt"></i> Sécurité du compte
                        </h2>
                        
                        <div class="form-grid">
                            <div class="form-group <?php echo isset($errors['username']) ? 'has-error' : ''; ?>">
                                <label for="username">Nom d'utilisateur *</label>
                                <input type="text" 
                                       id="username" 
                                       name="username" 
                                       required
                                       value="<?php echo htmlspecialchars($_POST['username'] ?? ''); ?>"
                                       placeholder="jean.dupont"
                                       oninput="checkUsername()">
                                <?php if (isset($errors['username'])): ?>
                                    <div class="error-message">
                                        <i class="fas fa-exclamation-circle"></i>
                                        <?php echo htmlspecialchars($errors['username']); ?>
                                    </div>
                                <?php endif; ?>
                                <div id="usernameFeedback" style="font-size: 12px; margin-top: 5px;"></div>
                            </div>
                            
                            <div class="form-group <?php echo isset($errors['password']) ? 'has-error' : ''; ?>">
                                <label for="password">Mot de passe *</label>
                                <div style="position: relative;">
                                    <input type="password" 
                                           id="password" 
                                           name="password" 
                                           required
                                           placeholder="Minimum 8 caractères"
                                           oninput="checkPasswordStrength()">
                                    <button type="button" class="toggle-password" 
                                            onclick="togglePasswordVisibility('password')"
                                            style="position: absolute; right: 10px; top: 50%; transform: translateY(-50%);
                                                   background: none; border: none; color: #666; cursor: pointer;">
                                        <i class="fas fa-eye"></i>
                                    </button>
                                </div>
                                <div class="password-strength" id="passwordStrength">
                                    <div class="password-strength-bar"></div>
                                </div>
                                <div class="password-requirements" id="passwordRequirements">
                                    <div class="requirement unmet" id="reqLength">
                                        <i class="fas fa-circle"></i> 8 caractères minimum
                                    </div>
                                    <div class="requirement unmet" id="reqLower">
                                        <i class="fas fa-circle"></i> 1 minuscule
                                    </div>
                                    <div class="requirement unmet" id="reqUpper">
                                        <i class="fas fa-circle"></i> 1 majuscule
                                    </div>
                                    <div class="requirement unmet" id="reqNumber">
                                        <i class="fas fa-circle"></i> 1 chiffre
                                    </div>
                                </div>
                                <?php if (isset($errors['password'])): ?>
                                    <div class="error-message">
                                        <i class="fas fa-exclamation-circle"></i>
                                        <?php echo htmlspecialchars($errors['password']); ?>
                                    </div>
                                <?php endif; ?>
                            </div>
                            
                            <div class="form-group <?php echo isset($errors['confirm_password']) ? 'has-error' : ''; ?>">
                                <label for="confirm_password">Confirmer le mot de passe *</label>
                                <div style="position: relative;">
                                    <input type="password" 
                                           id="confirm_password" 
                                           name="confirm_password" 
                                           required
                                           placeholder="Répétez votre mot de passe"
                                           oninput="checkPasswordMatch()">
                                    <button type="button" class="toggle-password" 
                                            onclick="togglePasswordVisibility('confirm_password')"
                                            style="position: absolute; right: 10px; top: 50%; transform: translateY(-50%);
                                                   background: none; border: none; color: #666; cursor: pointer;">
                                        <i class="fas fa-eye"></i>
                                    </button>
                                </div>
                                <div id="passwordMatch" style="font-size: 12px; margin-top: 5px;"></div>
                                <?php if (isset($errors['confirm_password'])): ?>
                                    <div class="error-message">
                                        <i class="fas fa-exclamation-circle"></i>
                                        <?php echo htmlspecialchars($errors['confirm_password']); ?>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                        
                        <div style="margin-top: 30px;">
                            <button type="button" class="btn" onclick="prevStep(2)" 
                                    style="background: #95a5a6;">
                                <i class="fas fa-arrow-left"></i> Retour
                            </button>
                            <button type="button" class="btn btn-next" onclick="nextStep(4)" 
                                    style="float: right; background: #3498db;">
                                <i class="fas fa-arrow-right"></i> Suivant
                            </button>
                        </div>
                    </div>
                    
                    <!-- Étape 4 : Confirmation et conditions -->
                    <div class="register-section" id="section4">
                        <h2 style="color: #2c3e50; margin-bottom: 30px;">
                            <i class="fas fa-file-contract"></i> Conditions d'utilisation
                        </h2>
                        
                        <div style="background: #f8f9fa; padding: 20px; border-radius: 8px; margin-bottom: 30px; max-height: 300px; overflow-y: auto;">
                            <h3 style="color: #2c3e50; margin-bottom: 15px;">Conditions Générales d'Utilisation</h3>
                            
                            <p><strong>Article 1 : Objet</strong></p>
                            <p>Les présentes conditions générales d'utilisation (CGU) ont pour objet de définir les modalités de mise à disposition des services de la plateforme Blockchain Logistique et les conditions d'utilisation du service par l'utilisateur.</p>
                            
                            <p><strong>Article 2 : Accès au service</strong></p>
                            <p>Le service est accessible gratuitement à tout utilisateur disposant d'un accès à Internet. Tous les frais supportés par l'utilisateur pour accéder au service (matériel informatique, logiciels, connexion Internet, etc.) sont à sa charge.</p>
                            
                            <p><strong>Article 3 : Collecte des données</strong></p>
                            <p>Dans le cadre de l'utilisation de la plateforme, l'éditeur est susceptible de collecter les données suivantes concernant l'utilisateur :</p>
                            <ul>
                                <li>Données d'identification (nom, prénom, adresse email)</li>
                                <li>Données professionnelles (nom de l'entreprise, rôle)</li>
                                <li>Données de connexion (adresse IP, jour et heure de connexion)</li>
                                <li>Données de traçabilité (informations sur les expéditions)</li>
                            </ul>
                            
                            <p><strong>Article 4 : Utilisation des données</strong></p>
                            <p>Les données collectées auprès des utilisateurs ont pour objet :</p>
                            <ul>
                                <li>La fourniture des services de la plateforme</li>
                                <li>L'amélioration de la qualité des services</li>
                                <li>La communication d'informations relatives aux services</li>
                                <li>La traçabilité blockchain des transactions</li>
                            </ul>
                            
                            <p><strong>Article 5 : Propriété intellectuelle</strong></p>
                            <p>Les marques, logos, signes et tout autre contenu de la plateforme font l'objet d'une protection par le Code de la propriété intellectuelle et plus particulièrement par le droit d'auteur.</p>
                            
                            <p><strong>Article 6 : Responsabilité</strong></p>
                            <p>L'éditeur s'engage à mettre en œuvre tous les moyens nécessaires pour garantir la sécurité et la confidentialité des données. Cependant, il n'apporte aucune garantie d'absence totale d'erreur, d'omission ou d'interruption de service.</p>
                            
                            <p><strong>Article 7 : Évolution des conditions</strong></p>
                            <p>Les présentes conditions générales d'utilisation peuvent être modifiées ou complétées à tout moment. Les utilisateurs seront informés de ces modifications.</p>
                            
                            <p style="margin-top: 20px; font-size: 12px; color: #666;">
                                Dernière mise à jour : <?php echo date('d/m/Y'); ?>
                            </p>
                        </div>
                        
                        <div style="margin: 30px 0;">
                            <label style="display: flex; align-items: flex-start; gap: 10px; cursor: pointer;">
                                <input type="checkbox" 
                                       name="terms" 
                                       id="terms"
                                       <?php echo isset($_POST['terms']) ? 'checked' : ''; ?>
                                       style="margin-top: 3px;">
                                <div>
                                    <div style="font-weight: 600; color: #2c3e50;">
                                        J'accepte les conditions générales d'utilisation *
                                    </div>
                                    <div style="font-size: 14px; color: #666; margin-top: 5px;">
                                        En cochant cette case, vous reconnaissez avoir lu, compris et accepté nos conditions d'utilisation et notre politique de confidentialité.
                                    </div>
                                </div>
                            </label>
                            <?php if (isset($errors['terms'])): ?>
                                <div class="error-message" style="margin-top: 10px;">
                                    <i class="fas fa-exclamation-circle"></i>
                                    <?php echo htmlspecialchars($errors['terms']); ?>
                                </div>
                            <?php endif; ?>
                        </div>
                        
                        <div style="margin-top: 30px;">
                            <label style="display: flex; align-items: center; gap: 10px; cursor: pointer; margin-bottom: 20px;">
                                <input type="checkbox" name="newsletter" id="newsletter" checked>
                                <span style="color: #666;">
                                    Je souhaite recevoir les newsletters et informations sur les nouvelles fonctionnalités
                                </span>
                            </label>
                        </div>
                        
                        <div class="register-footer">
                            <button type="button" class="btn" onclick="prevStep(3)" 
                                    style="background: #95a5a6;">
                                <i class="fas fa-arrow-left"></i> Retour
                            </button>
                            
                            <button type="submit" class="btn btn-success" 
                                    style="background: linear-gradient(135deg, #27ae60 0%, #229954 100%);">
                                <i class="fas fa-check-circle"></i> Créer mon compte
                            </button>
                        </div>
                    </div>
                </form>
                
                <div style="text-align: center; margin-top: 40px; padding-top: 20px; border-top: 1px solid #eee;">
                    <p style="color: #666;">
                        Déjà un compte ? 
                        <a href="login.php" style="color: #3498db; text-decoration: none; font-weight: 600;">
                            <i class="fas fa-sign-in-alt"></i> Se connecter
                        </a>
                    </p>
                </div>
            </div>
        </div>
    </div>
    
    <script src="js/main.js"></script>
    <script>
        // Gestion des étapes
        let currentStep = 1;
        
        function showStep(stepNumber) {
            // Masquer toutes les sections
            document.querySelectorAll('.register-section').forEach(section => {
                section.classList.remove('active');
            });
            
            // Désactiver toutes les étapes
            document.querySelectorAll('.step').forEach(step => {
                step.classList.remove('active', 'completed');
            });
            
            // Activer la section correspondante
            document.getElementById('section' + stepNumber).classList.add('active');
            
            // Activer et marquer les étapes précédentes comme complétées
            for (let i = 1; i <= stepNumber; i++) {
                const step = document.getElementById('step' + i);
                if (i === stepNumber) {
                    step.classList.add('active');
                } else {
                    step.classList.add('completed');
                }
            }
            
            currentStep = stepNumber;
        }
        
        function nextStep(nextStepNumber) {
            // Valider l'étape actuelle avant de passer à la suivante
            if (validateStep(currentStep)) {
                showStep(nextStepNumber);
                window.scrollTo({ top: 0, behavior: 'smooth' });
            }
        }
        
        function prevStep(prevStepNumber) {
            showStep(prevStepNumber);
            window.scrollTo({ top: 0, behavior: 'smooth' });
        }
        
        function validateStep(stepNumber) {
            let isValid = true;
            
            switch (stepNumber) {
                case 1:
                    // Valider les informations personnelles
                    const name = document.getElementById('name').value.trim();
                    const email = document.getElementById('email').value.trim();
                    const company = document.getElementById('company').value.trim();
                    
                    if (!name) {
                        markInvalid('name', 'Le nom est obligatoire');
                        isValid = false;
                    } else {
                        markValid('name');
                    }
                    
                    if (!email) {
                        markInvalid('email', 'L\'email est obligatoire');
                        isValid = false;
                    } else if (!validateEmail(email)) {
                        markInvalid('email', 'Format d\'email invalide');
                        isValid = false;
                    } else {
                        markValid('email');
                    }
                    
                    if (!company) {
                        markInvalid('company', 'L\'entreprise est obligatoire');
                        isValid = false;
                    } else {
                        markValid('company');
                    }
                    break;
                    
                case 2:
                    // Valider le rôle
                    const roleSelected = document.querySelector('input[name="role"]:checked');
                    if (!roleSelected) {
                        showAlert('Erreur', 'Veuillez sélectionner un rôle');
                        isValid = false;
                    }
                    break;
                    
                case 3:
                    // Valider la sécurité
                    const username = document.getElementById('username').value.trim();
                    const password = document.getElementById('password').value;
                    const confirmPassword = document.getElementById('confirm_password').value;
                    
                    if (!username) {
                        markInvalid('username', 'Le nom d\'utilisateur est obligatoire');
                        isValid = false;
                    } else if (username.length < 3) {
                        markInvalid('username', 'Minimum 3 caractères');
                        isValid = false;
                    } else if (!/^[a-zA-Z0-9_]+$/.test(username)) {
                        markInvalid('username', 'Caractères autorisés : lettres, chiffres et underscore');
                        isValid = false;
                    } else {
                        markValid('username');
                    }
                    
                    if (!password) {
                        markInvalid('password', 'Le mot de passe est obligatoire');
                        isValid = false;
                    } else if (password.length < 8) {
                        markInvalid('password', 'Minimum 8 caractères');
                        isValid = false;
                    } else if (!/(?=.*[a-z])(?=.*[A-Z])(?=.*\d)/.test(password)) {
                        markInvalid('password', 'Majuscule, minuscule et chiffre requis');
                        isValid = false;
                    } else {
                        markValid('password');
                    }
                    
                    if (!confirmPassword) {
                        markInvalid('confirm_password', 'La confirmation est obligatoire');
                        isValid = false;
                    } else if (password !== confirmPassword) {
                        markInvalid('confirm_password', 'Les mots de passe ne correspondent pas');
                        isValid = false;
                    } else {
                        markValid('confirm_password');
                    }
                    break;
            }
            
            return isValid;
        }
        
        // Sélection du rôle
        function selectRole(role) {
            // Désélectionner toutes les cartes
            document.querySelectorAll('.role-card').forEach(card => {
                card.classList.remove('selected');
                card.querySelector('input[type="radio"]').checked = false;
            });
            
            // Sélectionner la carte cliquée
            const selectedCard = document.querySelector(`.role-card[onclick*="${role}"]`);
            if (selectedCard) {
                selectedCard.classList.add('selected');
                selectedCard.querySelector('input[type="radio"]').checked = true;
            }
        }
        
        // Vérification du nom d'utilisateur
        function checkUsername() {
            const username = document.getElementById('username').value.trim();
            const feedback = document.getElementById('usernameFeedback');
            
            if (!username) {
                feedback.textContent = '';
                feedback.style.color = '#666';
                return;
            }
            
            // Vérifier la longueur
            if (username.length < 3) {
                feedback.textContent = 'Minimum 3 caractères';
                feedback.style.color = '#e74c3c';
                return;
            }
            
            // Vérifier les caractères autorisés
            if (!/^[a-zA-Z0-9_]+$/.test(username)) {
                feedback.textContent = 'Caractères autorisés : lettres, chiffres, underscore';
                feedback.style.color = '#e74c3c';
                return;
            }
            
            // Simuler une vérification d'existence
            const existingUsers = ['transporteur_1', 'producteur_1', 'destinataire_1', 'auditeur_1'];
            if (existingUsers.includes(username)) {
                feedback.textContent = 'Ce nom d\'utilisateur est déjà pris';
                feedback.style.color = '#e74c3c';
                return;
            }
            
            feedback.textContent = 'Nom d\'utilisateur disponible';
            feedback.style.color = '#27ae60';
        }
        
        // Vérification de la force du mot de passe
        function checkPasswordStrength() {
            const password = document.getElementById('password').value;
            const strengthBar = document.getElementById('passwordStrength');
            const requirements = {
                length: document.getElementById('reqLength'),
                lower: document.getElementById('reqLower'),
                upper: document.getElementById('reqUpper'),
                number: document.getElementById('reqNumber')
            };
            
            let score = 0;
            const totalRequirements = 4;
            
            // Vérifier les exigences
            if (password.length >= 8) {
                requirements.length.classList.remove('unmet');
                requirements.length.classList.add('met');
                score++;
            } else {
                requirements.length.classList.remove('met');
                requirements.length.classList.add('unmet');
            }
            
            if (/[a-z]/.test(password)) {
                requirements.lower.classList.remove('unmet');
                requirements.lower.classList.add('met');
                score++;
            } else {
                requirements.lower.classList.remove('met');
                requirements.lower.classList.add('unmet');
            }
            
            if (/[A-Z]/.test(password)) {
                requirements.upper.classList.remove('unmet');
                requirements.upper.classList.add('met');
                score++;
            } else {
                requirements.upper.classList.remove('met');
                requirements.upper.classList.add('unmet');
            }
            
            if (/[0-9]/.test(password)) {
                requirements.number.classList.remove('unmet');
                requirements.number.classList.add('met');
                score++;
            } else {
                requirements.number.classList.remove('met');
                requirements.number.classList.add('unmet');
            }
            
            // Mettre à jour la barre de force
            strengthBar.className = 'password-strength';
            if (score === 0) {
                // Aucune exigence remplie
            } else if (score <= 2) {
                strengthBar.classList.add('weak');
            } else if (score === 3) {
                strengthBar.classList.add('medium');
            } else {
                strengthBar.classList.add('strong');
            }
        }
        
        // Vérification de la correspondance des mots de passe
        function checkPasswordMatch() {
            const password = document.getElementById('password').value;
            const confirmPassword = document.getElementById('confirm_password').value;
            const matchFeedback = document.getElementById('passwordMatch');
            
            if (!password || !confirmPassword) {
                matchFeedback.textContent = '';
                return;
            }
            
            if (password === confirmPassword) {
                matchFeedback.textContent = 'Les mots de passe correspondent';
                matchFeedback.style.color = '#27ae60';
            } else {
                matchFeedback.textContent = 'Les mots de passe ne correspondent pas';
                matchFeedback.style.color = '#e74c3c';
            }
        }
        
        // Basculer la visibilité du mot de passe
        function togglePasswordVisibility(fieldId) {
            const input = document.getElementById(fieldId);
            const button = input.nextElementSibling;
            const icon = button.querySelector('i');
            
            if (input.type === 'password') {
                input.type = 'text';
                icon.className = 'fas fa-eye-slash';
            } else {
                input.type = 'password';
                icon.className = 'fas fa-eye';
            }
        }
        
        // Validation de l'email
        function validateEmail(email) {
            const re = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
            return re.test(email);
        }
        
        // Fonctions utilitaires
        function markInvalid(fieldId, message) {
            const input = document.getElementById(fieldId);
            const formGroup = input.closest('.form-group');
            
            formGroup.classList.add('has-error');
            
            // Ajouter ou mettre à jour le message d'erreur
            let errorMessage = formGroup.querySelector('.error-message');
            if (!errorMessage) {
                errorMessage = document.createElement('div');
                errorMessage.className = 'error-message';
                formGroup.appendChild(errorMessage);
            }
            
            errorMessage.innerHTML = `<i class="fas fa-exclamation-circle"></i> ${message}`;
        }
        
        function markValid(fieldId) {
            const input = document.getElementById(fieldId);
            const formGroup = input.closest('.form-group');
            
            formGroup.classList.remove('has-error');
            
            // Supprimer le message d'erreur
            const errorMessage = formGroup.querySelector('.error-message');
            if (errorMessage) {
                errorMessage.remove();
            }
        }
        
        function showAlert(title, message, type = 'error') {
            const alertDiv = document.createElement('div');
            alertDiv.className = `alert-notification ${type}`;
            alertDiv.innerHTML = `
                <div class="alert-icon ${type}">
                    <i class="fas fa-${type === 'success' ? 'check-circle' : 'exclamation-triangle'}"></i>
                </div>
                <div class="alert-content">
                    <div class="alert-title">${title}</div>
                    <div class="alert-message">${message}</div>
                </div>
                <button class="alert-close" onclick="this.parentElement.remove()">&times;</button>
            `;
            
            document.body.appendChild(alertDiv);
            
            setTimeout(() => {
                if (alertDiv.parentElement) {
                    alertDiv.remove();
                }
            }, 5000);
        }
        
        // Initialiser la sélection du rôle si déjà choisie
        document.addEventListener('DOMContentLoaded', function() {
            const selectedRole = "<?php echo $_POST['role'] ?? ''; ?>";
            if (selectedRole) {
                selectRole(selectedRole);
            }
            
            // Initialiser la vérification du mot de passe
            checkPasswordStrength();
            checkPasswordMatch();
            checkUsername();
        });
    </script>
</body>
</html>